/*
  AR-40D Rotator Controller
  written by Glen Popiel - KW5GP
*/

// #define debug  // Enables Debug Mode - Sends Debug output to Serial Monitor
// #define debug1 // Sends adc debug data to Serial Monitor
// # define debug2  // Sends switch debug data to Serial Monitor

#include <Adafruit_GFX.h>    // Core graphics library
#include <Adafruit_ST7735.h> // Hardware-specific library
#include <SPI.h>
#include <EEPROM.h>  // Include EEPROM Library
#include <ADS1115.h>

const int comm_speed = 19200;  // Set the Serial Monitor Baud Rate

#define rotate_right 10  // Assign Rotate Switch right input to Pin 10
#define rotate_left 2  // Assign Rotate Switch right input to Pin 2
#define left 3   // Assign Left (Counter Clockwise) Relay to Pin 3
#define right 4  // Assign Right (Clockwise) Relay to Pin 4
#define cal_zero 9    // Assign Zero Calibrate switch to Pin 9
#define cal_360 5     // Assign 360 Calibrate Switch to Pin 5
#define start_switch A1 // Assign the Start switch to pin A1
#define dial_pot A0 // Assign the Dial pot to pin A0
#define TFT_CS     6  // Assign the TFT CS to pin 6
#define TFT_RST    7  // Assign the TFT RST to pin 7
#define TFT_DC     8  // Assign the TFT DC to pin 8

#define tft_delay 10  // set the TFT command delay to 10ms
#define debounce 10  // set the switch debounce delay to 10ms


#define EEPROM_ID_BYTE 1   // EEPROM ID to validate EEPROM data location
#define EEPROM_ID  54  // EEPROM ID Value
#define EEPROM_AZ_CAL_0 2    // Azimuth Zero Calibration EEPROM location     
#define EEPROM_AZ_CAL_MAX 4  // Azimuth Max Calibration Data EEPROM location  

#define AZ_CAL_0_DEFAULT 30 // Set the default CAL zero point to 30
#define AZ_CAL_MAX_DEFAULT 25000 // Set the default CAL MAX point to 25000

#define AZ_Tolerance 1  // Set the Azimuth Accuracy Tolerance

int long current_AZ;  // Variable for current Azimuth ADC Value
int AZ_Degrees;  // Variable for Current Azimuth in Degrees
boolean moving = false;  // Variable to let us know if the rotor is moving
int long previous_AZ = -1;  // Variable to track the previous AZ reading
String Azimuth = "   ";
boolean calibrate = false;  // Variable used to determine if calibrate switch has been pressed
boolean first_pass = true;  // Variable used to determine if first pass through the main loop

// A/D Corrections for Position Sensor when not moving
const int  AZ_Correction[12] = { -550, -1500, -2100, -1900, -2350, -2400, -2800, -2500, -2000 , -1850, -1000, 27};

// A/D Corrections for Position Sensor when moving right (CW)
const int  R_correct[12] = { -700, -1600, -2300, -2100, -2500, -2600, -3000, -2700, -2200 , -2000, -1300, -50};

// A/D Corrections for Position Sensor when moving left (CCW)
const int  L_correct[12] = { -550, -1350, -1900, -1700, -2200, -2200, -2600, -2300, -1800 , -1730, -700, -900};

const int r_zero_adj = 0; // A/D zero point correction when moving right
const int l_zero_adj = -210; // A/D zero point correction when moving left
const int r_max_adj = 0; // A/D max point correction when moving right
const int l_max_adj = 0; // A/D max point correction when moving right

const int AZ_Max_Correction = 30; // Correction to reduce MAX cal point to prevent hitting stop

int set_AZ;  // Azimuth set value
int AZ_0;  // Azimuth Zero Value from EEPROM
int AZ_MAX; // Azimuth Max Value from EEPROMs
int zone_size; // Approximate number of A/D counts per zone
int zone = 0; // Current rotation zone

int long update_time = 0; // Set the Display update time counter to zero
int update_delay = 1000; // Set the Display update delay to one second
boolean turn_signal = false;  // Turn off the turn signl indicator
String previous_direction = "S";  // Set the previous direction to "S" - stop

String rotate_direction;  // Variable for the rotation direction

byte inByte = 0;  // incoming serial byte
byte serial_buffer[50];  // incoming serial byte buffer
int serial_buffer_index = 0;  // The index pointer variable for the Serial buffer
String Serial_Send_Data; // Data to send to Serial Port
String Requested_AZ; // RS232 Requested Azimuth - M and short W command
int AZ_To; // Requested AZ Move
int AZ_Distance; // Distance to move AZ

boolean manual_move = false;  // Variable to indicate this is a manually inititiated move
boolean right_status; // Variable to indicate the move right (CW) switch status
boolean left_status; // Variable to indicate the move left (CCW) switch status
boolean start_status; // Variable to indicate the Start switch status
boolean start_move; // Variable to indicate that a move has been started
boolean start_display = false;  // Variable to indicate the status of the display for a manual move

int previous_dial_pos = -100; // Variable for the previous dial pot position
int previous_dial_degrees = -100; // Variable for the previous dial pot position in degrees
int dial_degrees; // Variable for the current dial position in degrees
int dial_pos; // Variable for the current dial position
String dial_display = "   ";  // Variable for the dial display string
String dial_rotate = "   ";  // Variable for the dial rotating string

ADS1115 adc;  // Define the A/D as adc

Adafruit_ST7735 tft = Adafruit_ST7735(TFT_CS,  TFT_DC, TFT_RST);  // Initialize the TFT display

void setup()
{
  // initialize the digital pins
  pinMode(right, OUTPUT);
  pinMode(left, OUTPUT);
  pinMode(cal_zero, INPUT_PULLUP);
  pinMode(cal_360, INPUT_PULLUP);
  pinMode(rotate_right, INPUT_PULLUP);
  pinMode(rotate_left, INPUT_PULLUP);
  pinMode(start_switch, INPUT_PULLUP);

  // Turn off the rotator control relays
  digitalWrite(right, LOW);
  digitalWrite(left, LOW);

  Wire.begin(); // Join the I2C Bus

  adc.initialize(); // Initialize the ADS1115 16 bit A/D module
  Wire.beginTransmission(0x48); // Begin direct communication with ADC
  Wire.write(0x1);  // Connect to the ADC and send two bytes - Set the config register to all 1's
  Wire.write(0x7F); // MSB
  Wire.write(0xFF); // LSB
  Wire.endTransmission(); // End the direct ADC comms

  adc.setMode(ADS1115_MODE_CONTINUOUS); // free running conversion
  adc.setGain(ADS1115_PGA_2P048);  // set adc gain to 2.048v range, .0625 mv/step
  adc.setRate(ADS1115_RATE_128);  // set adc sample rate to 128 samples per second
  adc.setMultiplexer(ADS1115_MUX_P0_NG);  // AN0+ Vs ground - Single mode input on A0

  tft.initR(INITR_18BLACKTAB);   // initialize a 1.8" TFT with ST7735S chip, black tab
  delay(tft_delay);

  Serial.begin(comm_speed);  // Start the Serial port

  tft.fillScreen(ST7735_BLUE); // Clear the display
  delay(tft_delay);
  tft.setRotation(1); // Set the screen rotation
  delay(tft_delay);
  tft.setTextWrap(false);
  delay(tft_delay);
  tft.setTextSize(3);
  delay(tft_delay);
  tft.setTextColor(ST7735_GREEN);
  delay(tft_delay);
  tft.setCursor(40, 10);
  delay(tft_delay);
  tft.print("KW5GP");
  delay(tft_delay);
  tft.setTextSize(2);
  delay(tft_delay);
  tft.setCursor(55, 60);
  delay(tft_delay);
  tft.print("AR-40D");
  delay(tft_delay);
  tft.setCursor(45, 80);
  delay(tft_delay);
  tft.print("Rotator" );
  delay(tft_delay);
  tft.setCursor(25, 100);
  delay(tft_delay);
  tft.print("Controller");

  read_eeprom_cal_data();  // Read the EEPROM calibration data

#ifdef debug  // Display the calibration data when in debug mode
  Serial.print("ROM Cal - Zero: ");
  Serial.print(AZ_0);
  Serial.print(" Max: ");
  Serial.print(AZ_MAX);
#endif

  zone_size = AZ_MAX / 12;  // Divide the rotation position into 12 zones for calilbration correction
  AZ_MAX = AZ_MAX - AZ_Max_Correction;  // Subract a bit from the MAX position to keep from hitting the stop

#ifdef debug
  Serial.print(" Adj Max: ");
  Serial.print(AZ_MAX);
  Serial.print(" Zone Size: ");
  Serial.println(zone_size);
#endif

  delay(5000);  //Wait 5 seconds then clear the startup message
  tft.fillScreen(ST7735_BLUE); // Clear the display
  delay(tft_delay);

  set_AZ = -1;  // Preset the Azimuth Move Variable

  read_dial_pot(); // Read the dial pot

  update_display(); // update the display
  first_pass = false; // turn off the update display first pass flag

}  // End Setup Loop

void loop() // Start the Main Loop
{
  update_display(); // update the display
  check_serial(); // Check for a Serial command on the USB port
  check_move(); // Check to see if we are moving and process the move
  check_switches(); // Check the switches for manual commands

  // If not moving, read the dial pot and check if a calibration switch is pressed
  if (!moving)
  {
    read_dial_pot();
    check_cal();  // Check to see if either calibration switch is pressed
  }

#ifdef debug1
  delay(250);
#endif

}  // End Main Loop

// check_cal() function - Checks to see if a calibration switch is pressed and set calibration accordingly
void check_cal()
{
  if (digitalRead(cal_zero) == 0) // Cal Zero button pressed
  {
    set_0_az_cal(); // Set the Cal zero point
  }
  if (digitalRead(cal_360) == 0) // Cal Max button pressed
  {
    set_max_az_cal(); // Set the Cal max point
  }
}

// all_stop() function - turns off relays, delays 3 seconds then turns off Brake Time indicator
void all_stop()
{
  tft.setTextSize(2);
  delay(tft_delay);
  clear_top_line();
  delay(tft_delay);
  tft.setCursor(40, 3); // Display message on LCD
  delay(tft_delay);
  tft.print("Braking");
  delay(tft_delay);
  digitalWrite(right, LOW);  // Turn off CW Relay
  delay(tft_delay);
  digitalWrite(left, LOW);  // Turn off CCW Relay
  delay(tft_delay);
  moving = false;
  rotate_direction = "S";  // Set direction to S (Stop)
  delay(3000);
  clear_top_line();
  delay(tft_delay);
}

// read_eeprom_cal_data() function - Reads the EEPROM Calibration data
void read_eeprom_cal_data()
{
  if (EEPROM.read(EEPROM_ID_BYTE) == EEPROM_ID)   // Verify the EEPROM has valid data
  {

#ifdef debug // Display the Calibration data in debug mode
    Serial.print("Read EEPROM Cal Data Valid - AZ_CAL_0: ");
    Serial.print((EEPROM.read(EEPROM_AZ_CAL_0) * 256) + EEPROM.read(EEPROM_AZ_CAL_0 + 1), DEC);
    Serial.print("   AZ_CAL_MAX: ");
    Serial.println((EEPROM.read(EEPROM_AZ_CAL_MAX) * 256) + EEPROM.read(EEPROM_AZ_CAL_MAX + 1), DEC);
#endif

    AZ_0 = (EEPROM.read(EEPROM_AZ_CAL_0) * 256) + EEPROM.read(EEPROM_AZ_CAL_0 + 1); // Set the Zero degree Calibration Point
    AZ_MAX = (EEPROM.read(EEPROM_AZ_CAL_MAX) * 256) + EEPROM.read(EEPROM_AZ_CAL_MAX + 1); // Set the 360 degree Calibration Point

  } else
  { // EEPROM has no Calibration data - initialize eeprom to default values

#ifdef debug
    // Send status message in debug mode
    Serial.println("Read EEPROM Cal Data Invalid - set to defaults");
#endif

    AZ_0 = AZ_CAL_0_DEFAULT;  // Set the Calibration data to default values
    AZ_MAX = AZ_CAL_MAX_DEFAULT;
    write_eeprom_cal_data();  // Write the data to the EEPROM
  }
}

// write_eeprom_cal_data() function - Writes the Calibration data to the EEPROM
void write_eeprom_cal_data() // Write the Calibration data to the EEPROM
{
#ifdef debug
  Serial.println("Writing EEPROM Cal Data");  // Display status in debug mode
#endif

  EEPROM.write(EEPROM_ID_BYTE, EEPROM_ID); // Write the EEPROM ID to the EEPROM
  EEPROM.write(EEPROM_AZ_CAL_0, highByte(AZ_0)); // Write Zero Calibration Data High Order Byte
  EEPROM.write(EEPROM_AZ_CAL_0 + 1, lowByte(AZ_0)); // Write Zero Calibration Data Low Order Byte
  EEPROM.write(EEPROM_AZ_CAL_MAX, highByte(AZ_MAX)); // Write 360 Calibration Data High Order Byte
  EEPROM.write(EEPROM_AZ_CAL_MAX + 1, lowByte(AZ_MAX)); // Write 360 Calibration Data Low Order Byte
}

// check_serial()function - Checks for data on the Serial/USB port
void check_serial()
{
  if (Serial.available() > 0) // Get the Serial Data if available
  {
    inByte = Serial.read();  // Get the Serial Data

    // You may need to uncomment the following line if your PC software
    // will not communicate properly with the controller

    //   Serial.print(char(inByte));  // Echo back to the PC

    if (inByte == 10)  // ignore Line Feeds
    {
      return;
    }
    if (inByte != 13) // Add to buffer if not CR
    {
      serial_buffer[serial_buffer_index] = inByte;

#ifdef debug // Print the Character received if in Debug mode
      Serial.print("RX: ");
      Serial.print(" *** ");
      Serial.print(serial_buffer[serial_buffer_index]);
      Serial.println(" *** ");
#endif

      serial_buffer_index++;  // Increment the Serial Buffer pointer

    } else
    { // It's a Carriage Return, execute command

      if ((serial_buffer[0] > 96) && (serial_buffer[0] < 123))  //If first character of command is lowercase, convert to uppercase
      {
        serial_buffer[0] = serial_buffer[0] - 32;
      }

      switch (serial_buffer[0]) {  // Decode first character of command

        case 65:  // A Command - Stop the Azimuth Rotation

#ifdef debug
          Serial.println("A RX");
#endif

          az_rotate_stop();
          break;

        case 67:      // C - return current azimuth

#ifdef debug
          Serial.println("C RX ");
#endif

          send_current_az();  // Return Azimuth if C Command
          break;

        case 70:  // F - Set the Max Calibration

#ifdef debug
          Serial.println("F RX");
          Serial.println(serial_buffer_index);
#endif

          set_max_az_cal();  // F - Set the Max Azimuth Calibration
          break;

        case 76:  // L - Rotate Azimuth CCW

#ifdef debug
          Serial.println("L RX");
#endif

          rotate_az_ccw();  // Call the Rotate Azimuth CCW function
          break;

        case 77:  // M - Rotate to Set Point

#ifdef debug
          Serial.println("M RX");
#endif

          rotate_to();  // Call the Rotate to Set Point Command
          break;

        case 79:  // O - Set Zero Calibration

#ifdef debug
          Serial.println("O RX");
          Serial.println(serial_buffer_index);
#endif

          set_0_az_cal();  // O - Set the Azimuth Zero Calibration
          break;

        case 82:  // R - Rotate Azimuth CW

#ifdef debug
          Serial.println("R RX");
#endif

          rotate_az_cw();  // Call the Rotate Azimuth CW function
          break;

        case 83:  // S - Stop All Rotation

#ifdef debug
          Serial.println("S RX");
#endif

          az_rotate_stop();  // Call the Stop Azimith Rotation function
          break;

      }

      serial_buffer_index = 0;  // Clear the Serial Buffer and Reset the Buffer Index Pointer
      serial_buffer[0] = 0;
    }
  }
}

// check_move() function - Checks to see if we've been commanded to move
void check_move()
{
  if (set_AZ != -1)
  {
    // We're moving - check and stop as needed
    update_delay = 100; // Change the display update delay to 100ms
    read_adc(); // Read the ADC

    // Map the Azimuth to degrees

#ifdef debug
    Serial.print(" *** check_move *** AZ_To: ");
    Serial.print(AZ_To);
    Serial.print("  Zone: ");
    Serial.print(zone);
    Serial.print("  Zone Adj: ");
    if (rotate_direction == "R")
    {
      Serial.print(R_correct[zone]);
    } else
    {
      Serial.print(L_correct[zone]);
    }
    Serial.print(" Current_AZ: ");
    Serial.print(current_AZ);
    Serial.print(" AZ_Deg: ");
    Serial.print(AZ_Degrees);
#endif

    if (set_AZ != -1) // If we're moving
    {
      AZ_Distance = set_AZ - AZ_Degrees;  // Check how far we have to move

#ifdef debug
      Serial.print("  AZ_Distance: ");
      Serial.print(AZ_Distance);
#endif

      fix_180();  // Adjust for North Centering

#ifdef debug
      Serial.print("  Adj AZ_Distance: ");
      Serial.print(AZ_Distance);
      Serial.print("  Direction: ");
      Serial.println(rotate_direction);
#endif

      if (abs(AZ_Distance) <= AZ_Tolerance)  // No move needed if we're within the tolerance range
      {
        az_rotate_stop();  // Stop the Azimuth Rotation
        set_AZ = -1;  // Turn off the Azimuth Move Command
      } else
      { // Move Azimuth - figure out which way
        if (AZ_Distance > 0)   //We need to move CW
        {
          rotate_az_cw();  // Rotate CW if positive
        } else {
          rotate_az_ccw();  // Rotate CCW if negative
        }
      }
    }
  } else
  {
    if (!manual_move && !start_move)  // Reset the display update delay to one second if we're not moving
    {
      update_delay = 1000;
    }
  }
}

// send_current_az() function - Sends the Current Azimuth function
void send_current_az()
{
  read_adc();  // Read the ADC

#ifdef debug
  Serial.println();
  Serial.print("Deg: ");
  Serial.print(AZ_Degrees);
  Serial.print("  Return Value: ");
#endif

  // Send it back via serial/USB port
  Serial_Send_Data = "";
  if (AZ_Degrees < 100)  // pad with 0's if needed
  {
    Serial_Send_Data = "0";
  }
  if (AZ_Degrees < 10)
  {
    Serial_Send_Data = "00";
  }
  Serial_Send_Data = "+0" + Serial_Send_Data + String(AZ_Degrees);  // Send the Azimuth in Degrees
  Serial.println(Serial_Send_Data);  // Return value via serial/USB port
}

void set_max_az_cal() // Set the Max Azimuth Calibration function
{
#ifdef debug
  Serial.println("Cal Max AZ Function");
#endif

  calibrate = true;
  read_adc();  // Read the ADC

  // save current az value to EEPROM - Zero Calibration

#ifdef debug
  Serial.println(current_AZ);
#endif

  AZ_MAX = current_AZ;  // Set the Azimuth Maximum Calibration to Current Azimuth Reading
  write_eeprom_cal_data();  // Write the Calibration Data to EEPROM

#ifdef debug
  Serial.println("Max Azimuth Cal Complete");
#endif

  calibrate = false;
}

// rotate_az_ccw()function - Rotate Azimuth CCW
void rotate_az_ccw()
{
  digitalWrite(left, HIGH);  // Set the Rotate Left Pin High
  digitalWrite(right, LOW);  // Make sure the Rotate Right Pin is Low
  rotate_direction = "L";  // set the direction flag to "L" (Left - CCW)

  // Turn on the turn signal on the display
  if (!turn_signal || (previous_direction != rotate_direction))
  {
    tft.setTextSize(2);
    delay(tft_delay);
    clear_top_line();
    delay(tft_delay);
    tft.setCursor(3, 10); // display the left arrow on the LCD
    delay(tft_delay);
    tft.print("<==");
    delay(tft_delay);
    turn_signal = true;
    previous_direction = rotate_direction;
  }
  moving = true;  // Set the moving flag
}

// rotate_az_cw() function - Rotate Azimuth CW
void rotate_az_cw()
{
  digitalWrite(right, HIGH);    // Set the Rotate Right Pin High
  digitalWrite(left, LOW);    // Make sure the Rotate Left Pin Low
  rotate_direction = "R";  // set the direction flag to "R" (Right - CW)
  if (!turn_signal || (previous_direction != rotate_direction))

    // Turn on the turn signal on the display
  {
    tft.setTextSize(2);
    delay(tft_delay);
    clear_top_line();
    delay(tft_delay);
    tft.setCursor(120, 10); // display the left arrow on the LCD
    delay(tft_delay);
    tft.print("==>");
    delay(tft_delay);
    turn_signal = true;
    previous_direction = rotate_direction;
  }
  moving = true;  // Set the moving flag
}

// az_rotate_stop() function - Stop Azimuth Rotation
void az_rotate_stop()
{
  digitalWrite(right, LOW);  // Turn off the Rotate Right Pin
  digitalWrite(left, LOW);   // Turn off the Rotate Left Pin

  // update the display
  tft.setTextSize(2);
  delay(tft_delay);
  clear_top_line();
  delay(tft_delay);
  tft.setCursor(40, 3); // Display Braking message on LCD
  delay(tft_delay);
  tft.print("Braking");
  delay(tft_delay);
  set_AZ = -1;

  if (start_move) // If this is a move commanded by the start button
  {
    // Clear the moving display
    tft.fillRect(15, 91, 140, 8, ST7735_BLUE);
    delay(tft_delay);
  }
  turn_signal = false;
  start_move = false;
  start_display = false;
  delay(3000);
  clear_top_line(); // Clear the top line of the display
  moving = false; // Clear the moving flag
  rotate_direction = "S";  // Set direction to S (Stop)
  previous_direction = rotate_direction;
}

// rotate_to() function - Rotate to Set Point
void rotate_to()
{
#ifdef debug
  Serial.println("M Command -  rotate_to Function");
  Serial.print("  Chars RX: ");
  Serial.print(serial_buffer_index);
#endif

  // Decode Command - Format Mxxx - xxx = Degrees to Move to
  if (serial_buffer_index == 4)  // Verify the command is the proper length
  {
#ifdef debug
    Serial.print("  Value [1] to [3]: ");
#endif

    Requested_AZ = (String(char(serial_buffer[1])) + String(char(serial_buffer[2])) + String(char(serial_buffer[3]))) ;  // Decode the Azimuth Value
    AZ_To = (Requested_AZ.toInt()); // AZ Degrees to Move to as integer

    // Flush the buffer to allow the next command
    serial_buffer_index = 0;

#ifdef debug
    Serial.println(Requested_AZ);
    Serial.print("AZ_To: ");
    Serial.print(AZ_To);
#endif

    read_adc();  // Read the ADC

#ifdef debug
    Serial.print("  Zone: ");
    Serial.print(zone);
    Serial.print("  Current Deg: ");
    Serial.print(AZ_Degrees);
#endif

    AZ_Distance = AZ_To - AZ_Degrees;  // Figure out how far we have to move

#ifdef debug
    Serial.print("  AZ_Dist: ");
    Serial.print(AZ_Distance);
#endif

    fix_180();  // Correct for North Centering

    set_AZ = AZ_To; // Set the Azimuth to move to
    moving = true;  // Set the moving flag

#ifdef debug
    Serial.print("  Adj AZ_Dist: ");
    Serial.print(AZ_Distance);
#endif

    if (abs(AZ_Distance) <= AZ_Tolerance)  // No move needed if we're within the Tolerance Range
    {
      az_rotate_stop();  // Stop the Azimuth Rotation
      set_AZ = -1;  // Turn off the Move Command
    } else
    { // Move Azimuth - figure out which way
      if (AZ_Distance > 0)   //We need to move CW
      {

#ifdef debug
        Serial.println("  Turn right ");
#endif

        rotate_az_cw();  // If the distance is positive, move CW
      } else
      {

#ifdef debug
        Serial.println("  Turn left ");
#endif
        rotate_az_ccw();  // Otherwise, move counterclockwise
      }
    }
  }
}

// set_0_az_cal() function - Sets Azimuth Zero Calibration
void set_0_az_cal()
{

#ifdef debug
  Serial.println("Cal Zero Function");
#endif

  calibrate = true;
  read_adc();  // Read the ADC
  // save current Azimuth value to EEPROM - Zero Calibration

#ifdef debug
  Serial.println(current_AZ);
#endif

  AZ_0 = current_AZ;  // Set the Azimuth Zero Calibration to current position
  write_eeprom_cal_data();  // Write the Calibration Data to EEPROM

#ifdef debug
  Serial.println("Zero AZ Cal Complete");
#endif

  calibrate = false;
}

// fix_az_string() function - Converts azimuth to string and pad to 3 characters
void fix_az_string()
{
  Azimuth = "00" + String(AZ_Degrees);
  Azimuth = Azimuth.substring(Azimuth.length() - 3);
}

// update_display() function - Updates the TFT display
void update_display()
{
  if (first_pass) // Only do this the first time the function is called
  {
    // Clear the screen and display normal operation
    tft.fillScreen(ST7735_BLUE); // Clear the display
    delay(tft_delay);
    tft.setTextSize(1); // Set the text size to 1
    delay(tft_delay);
    tft.setTextColor(ST7735_GREEN); // Set the text color to green
    delay(tft_delay);
    tft.setTextSize(2); // Set the text size to 2
    delay(tft_delay);
    tft.setCursor(32, 110);
    delay(tft_delay);
    tft.print("Dial: ");  // Display the dial position template text
    delay(tft_delay);

    previous_AZ = -1;  // Reset the previous AZ before starting
    first_pass = false; // turn off the first pass flag
  }

  // Check to see if it's time to update the display data
  if (abs(millis()) > abs(update_time + update_delay))  // check to see if update time has expired
  {
    read_adc(); // Read the ADC
    if (AZ_Degrees != previous_AZ)  // If the ADC reading has change update the display
    {
      tft.setTextSize(6); // Set the text size to 6
      delay(tft_delay);
      tft.fillRect(30, 30, 130, 55, ST7735_BLUE); // Clear the position display area
      delay(tft_delay);
      tft.setTextColor(ST7735_GREEN); // Set the text color to green
      delay(tft_delay);
      tft.setCursor(30, 40);
      delay(tft_delay);
      tft.print(Azimuth); // Display the azimuth in degrees
      delay(tft_delay);
      tft.setCursor(140, 30);
      delay(tft_delay);
      tft.setTextSize(3); // Set the text size to 3
      delay(tft_delay);
      tft.print("o"); // display the degree symbol
      delay(tft_delay);
      previous_AZ = AZ_Degrees; // set the previous azimuth reading to the current reading

      // If it's the start of a manually commanded move update the display to relect this
      if (start_move && !start_display)
      {
        // It's a manual dial move
        tft.setTextSize(1);
        delay(tft_delay);
        tft.setCursor(15, 91);
        delay(tft_delay);
        tft.print("Moving to Dial Position");
        delay(tft_delay);
        start_display = true;
      }

    }
    if (previous_dial_degrees != dial_degrees)  // if the dial pot position has changed
    {
      tft.fillRect(95, 110, 40, 20, ST7735_BLUE); // Clear the dial pot display area
      delay(tft_delay);
      tft.setCursor(95, 110);
      delay(tft_delay);
      tft.setTextSize(2); // Set the text size to 2
      delay(tft_delay);
      tft.print(dial_display);  // Display the current dial pot position
      delay(tft_delay);
      tft.setTextSize(1); // Set the text size to 1
      delay(tft_delay);
      tft.setCursor(130, 105);
      delay(tft_delay);
      tft.print("o"); // display the degree symbol
      delay(tft_delay);
      previous_dial_degrees = dial_degrees; // set the previous dial reading to the current reading
    }
    update_time = millis(); // Reset the display update timer
  }

#ifdef debug1
  Serial.print("    ");
  Serial.print(" Deg: ");
  Serial.println(AZ_Degrees); // Send position to Serial Monitor in debug mode
#endif

}

// read_adc() function - Reads the A/D Converter
void read_adc()
{
  // Read ADC and display position
  adc.setGain(ADS1115_PGA_2P048);  // set adc gain to 2.048v range, .0625 mv/step
  adc.setRate(ADS1115_RATE_128);  // set adc sample rate to 128 samples per second

  // Read adc
  current_AZ = adc.getDiff0();  // Read ADC channel 0

#ifdef debug1
  Serial.print("   Read ADC: "); // display ADC read status in debug mode
  Serial.print(current_AZ);  // Display ADC value in debug mode
#endif

  if (calibrate)  // Exit if we're calibrating
  {
    return;
  }

  if (current_AZ < 0) // Force the reading to zero if negative
  {
    current_AZ = 0;
  }

  if (current_AZ > AZ_MAX)  // Force the reading to max if we're above max
  {
    current_AZ = AZ_MAX;
  }

  zone_correct(); // Adjust A/D based on rotation zone

  map_az(); // Map the azimuth to degrees

  fix_az_string();  // Convert azimuth to string then pad to 3 characters and save in Azimuth

#ifdef debug1
  Serial.print(" Moving: "); 
  Serial.print(moving);
  Serial.print(" set_AZ: "); 
  Serial.print(set_AZ);
#endif

}

// clear_top_line() function - Clears the top line of the display
void clear_top_line()
{
  tft.fillRect(0, 0, 160, 30, ST7735_BLUE);
  delay(tft_delay);
}


// zone_correct() function - Adjusts A/D value based on rotation zone
void zone_correct()
{
  int start_zone; // Variable to hold the start of the zone
  int end_zone; // Variable to hold the end of the zone
  int correction; // Variable to hold the correction value
  int start_pos;  // Start of current zone
  int end_pos;  // End of current zone
  int current_pos;  // current position in zone
  int calculated_AZ;  // Used to adjust current AZ at zero point

  zone = current_AZ / zone_size;  // Figure out what rotation zone we're in

  if (zone < 0) // Force the zone to zero if below zero
  {
    zone = 0;
  }
  if (zone > 11)  // Force the zone to 11 if we're above 11
  {
#ifdef debug1
    Serial.println(" Zone forced to 11");
#endif
    zone = 11;
  }

  start_pos = (zone * zone_size); // Calulate the start of the zone
  end_pos = start_pos + zone_size;  // Calculate the end of the zone

#ifdef debug1
  Serial.print("  Zone: ");
  Serial.print(zone);
#endif

  // Handle Zone zero differently
  if (zone == 0)
  {
    start_zone = 0;
    if (!moving)  // If we're not moving, use static correction values
    {
      start_pos = AZ_0;
      end_zone = AZ_Correction[0];
    } else
    {
      // we're moving - use moving correction factors
      if (rotate_direction == "R")
      {
        // We're turning right
        start_pos = AZ_0;
        end_zone = R_correct[0];
        current_AZ = current_AZ - r_zero_adj ;
      } else
      {
        // We're turning left
        start_pos = AZ_0;
        end_zone = L_correct[0];
        current_AZ = current_AZ - l_zero_adj ;
      }
    }
  }

  // Handle zone 11 differently
  if (zone == 11)
  {
    if (moving)
    {
      // we're moving - use moving correction factors
      if (rotate_direction == "R")
      {
        start_zone = R_correct[zone - 1];
        end_zone = R_correct[zone];
        current_AZ = current_AZ  - r_max_adj;
      } else
      {
        // We're turning left - use left corrections
        end_pos = AZ_MAX;
        start_zone = L_correct[11];
        end_zone = l_max_adj;
        current_AZ = current_AZ  - l_max_adj;
      }
    } else
    {
      start_zone = AZ_Correction[zone - 1];
      end_zone = AZ_Correction[zone];
      calculated_AZ = current_AZ - AZ_0;
    }
  }

  calculated_AZ = current_AZ;

  // Handle zones 1 thru 10
  if (zone > 0 && zone < 11)
    if (moving)
    {
      // Add the moving correction value when moving
      if (rotate_direction == "R")
      {
        start_zone = AZ_Correction[zone - 1];
        end_zone = R_correct[zone];
      } else
      {
        // We're turning left - use left corrections
        start_zone = AZ_Correction[zone - 1];
        end_zone =  L_correct[zone];
      }
    } else
    {
      start_zone = AZ_Correction[zone - 1];
      end_zone = AZ_Correction[zone];
      calculated_AZ = current_AZ - AZ_0;
    }

#ifdef debug1
  Serial.print("  Zone data: ");
  Serial.print(start_zone);
  Serial.print(" ");
  Serial.print(end_zone);
  Serial.print("  Zone range: ");
  Serial.print(start_pos);
  Serial.print(" ");
  Serial.print(end_pos);
#endif

  // Map the correction factor based on position in zone

  correction = map(calculated_AZ, start_pos, end_pos, start_zone, end_zone); // Map the correction

  current_AZ = current_AZ + correction;

  if (current_AZ < 0) // Force the current azimuth value to zero if negative
  {
    current_AZ = 0;
  }

#ifdef debug1
  Serial.print(" Adj: ");
  Serial.print(correction);
  Serial.print(" Final AZ: ");
  Serial.print(current_AZ);
#endif

}

// map_az()function - maps azimuth value to degrees
void map_az()
{
  AZ_Degrees = map(current_AZ, AZ_0, AZ_MAX, 0, 360); // Map the current_AZ to calibrated degrees

  if (AZ_Degrees < 0) // Limit Zero point to 0
  {
    AZ_Degrees = 0;
  }

  if (AZ_Degrees > 360) // Limit max point to 360
  {
    AZ_Degrees = 360;
  }

  if (AZ_Degrees < 180)  // Adjust for North-centered controller
  {
    AZ_Degrees = 180 + AZ_Degrees;
  } else
  {
    AZ_Degrees = AZ_Degrees - 180;
  }

}

// fix_180() function - Correct for North-centering
void fix_180()
{
  if (AZ_Degrees >= 180 && AZ_Degrees <= 360  && zone != 11) // Make sure we don't rotate left beyond 180 and we're not in zone 11
  {
    if ((AZ_Degrees + AZ_Distance < 180) && (AZ_Distance < 0))
    {
      AZ_Distance = 360 + AZ_Distance;
    }
  } else {
    // We're in zone 4 thru 12 - Make sure we don't rotate right beyond 180

#ifdef debug
    Serial.print(" In Zone 4 thru 12");
#endif

    if ((AZ_Degrees + AZ_Distance > 180) && (AZ_Distance > 0))
    {
      AZ_Distance = -360 + AZ_Distance;
    }
  }
}

// check_switches() function - checks the left, right, and start move switches
void check_switches()
{
  // Checks the Rotate and Start switches
  right_status = digitalRead(rotate_right); // Read the Right move switch
  left_status = digitalRead(rotate_left); // Read the Left move switch
  start_status = digitalRead(start_switch); // Read the Start move switch

  // If any of the move switches are pressed
  if (left_status == LOW || right_status == LOW || start_status == LOW)
  {
    // manual left, right or start active
    if (!manual_move) // If we're not already moving
    {
      update_delay = 100; // Set the display update delay to 100ms
      if (left_status == LOW) // If the Left switch is pressed
      {
        // We're moving left
        rotate_az_ccw();  // Rotate Left
        manual_move = true; // Set the manual move flag

#ifdef debug2
        Serial.println("Rotate Left pressed");
#endif

      }

      if (right_status == LOW)// If the Right switch is pressed
      {
        // We're moving right
        rotate_az_cw();  // Rotate Right
        manual_move = true; // Set the manual move flag
        
#ifdef debug2
        Serial.println("Rotate Right pressed");
#endif

      }

      if (start_status == LOW && !start_move) // If the Start switch is pressed and we're not already moving
      {
        // Start switch is pressed - we need to move to dial position
        // Fake a serial command and stuff the buffer with the Move command data
        dial_rotate = dial_display; // Set the position to rotate to
        // Determine which way to turn if we're going to 180 degrees - move to closest 180 position
        if (dial_rotate == "180")
        {
          if (dial_pos > 500)
          {
            // Force right turn no matter what
            dial_rotate = "179";
          } else
          {
            // Force left turn no matter what
            dial_rotate = "181";
          }
        }
        // Create the rotate_to function command string
        serial_buffer[0] = 'M';
        serial_buffer[1] = dial_rotate[0];
        serial_buffer[2] = dial_rotate[1];
        serial_buffer[3] = dial_rotate[2];
        serial_buffer_index = 4;
        start_move = true;  // Set the manual start command flag

#ifdef debug2
        Serial.print("Start switch pressed - moving to dial position - Degrees: ");
        Serial.print(dial_display);
        Serial.print("  Buffer: ");
        Serial.print(serial_buffer[0]);
        Serial.print(serial_buffer[1]);
        Serial.print(serial_buffer[2]);
        Serial.print(serial_buffer[3]);
        Serial.print("  Length: ");
        Serial.println(serial_buffer_index);
#endif

        rotate_to();  // Call the rotate_to function
      }
      delay(debounce);  // Delay for switch debounce
    }
  } else
  {
    // We're not moving (anymore)
    if (manual_move)
    {
      az_rotate_stop(); // call the rotate stop function
      manual_move = false;  // Turn off the manual move flag

#ifdef debug2
      Serial.println("Switch Released");
#endif

    }
  }
}

// read_dial_pot() function - Reads the dial pot position
void read_dial_pot()
{
  // Reads the pot and updates display if it has changed
  dial_pos = analogRead(dial_pot);
  if (dial_pos > (previous_dial_pos + 1) || dial_pos < (previous_dial_pos - 1)) // If the dial pot has changed
  {
    // Dial pot has moved
    dial_degrees = map(dial_pos, 0, 1023, 0, 360) + 180;  // Map the dial position to degrees
    if (dial_degrees > 359) // Adjust for North Centering
    {
      dial_degrees = dial_degrees - 360;
    }
    previous_dial_pos = dial_pos; // Update the previous dial position

    fix_dial_string();  // Pads the dial string to 3 characters

#ifdef debug2
    Serial.print("Dial Pot : ");
    Serial.print(dial_pos);
    Serial.print("  Dial Degrees : ");
    Serial.println(dial_degrees);
#endif

    update_display(); // Update the display
  }
}

// fix_dial_string() function - Converts azimuth to string and pads to 3 characters
void fix_dial_string()
{
  dial_display = "00" + String(dial_degrees);
  dial_display = dial_display.substring(dial_display.length() - 3);
}

