/*******************************************************************************
* File Name: PrISM.c
* Version 1.20
*
* Description:
*  This file provides the source code to the API for the PrISM Component.
*
* Note:
*  The PRiSM Component consists of a 8, 16, 24, 32 - bit PrISM with
*  depended on length Polynomial value and user selected Seed Value. 
*
********************************************************************************
* Copyright 2008-2009, Cypress Semiconductor Corporation.  All rights reserved.
* You may use this file only in accordance with the license, terms, conditions, 
* disclaimers, and limitations in the end user license agreement accompanying 
* the software package with which this file was provided.
*******************************************************************************/

#include "PrISM.h"

uint8 PrISM_initVar = 0;


/*******************************************************************************
* FUNCTION NAME:   PrISM_Start
********************************************************************************
*
* Summary:
*  The start function sets Polynomial, Seed and Pulse Density registers provided
*  by customizer. PrISM computation starts on rising edge of input clock.
*
* Parameters:  
*  void
*
* Return:  
*  void
*
*******************************************************************************/
void PrISM_Start(void) 
{
    if (PrISM_initVar == 0)
    {
       PrISM_initVar = 0x01u; 
       /* Writes Seed value, polynom value and density provided by customizer */
       PrISM_WriteSeed(PrISM_SEED);
       PrISM_WritePolynomial(PrISM_POLYNOM);
       PrISM_WritePulse0(PrISM_DENSITY0);
       PrISM_WritePulse1(PrISM_DENSITY1);

        /* Set FIFO0_CLR bit to use FIFO0 as a simple one-byte buffer*/
        #if (PrISM_RESOLUTION <= 8)      /* 8bit - PrISM */
            CY_SET_REG8(PrISM_AUX_CONTROL_PTR, PrISM_FIFO0_CLR);
        #elif (PrISM_RESOLUTION <= 16)   /* 16bit - PrISM */
            CY_SET_REG16(PrISM_AUX_CONTROL_PTR, PrISM_FIFO0_CLR | 
                                                           PrISM_FIFO0_CLR << 8);
        #elif (PrISM_RESOLUTION <= 24)   /* 24bit - PrISM */
            CY_SET_REG24(PrISM_AUX_CONTROL_PTR, PrISM_FIFO0_CLR | 
                                                           PrISM_FIFO0_CLR << 8 );
            CY_SET_REG24(PrISM_AUX_CONTROL2_PTR, PrISM_FIFO0_CLR );
        #else                                 /* 32bit - PrISM */
            CY_SET_REG32(PrISM_AUX_CONTROL_PTR, PrISM_FIFO0_CLR | 
                                                           PrISM_FIFO0_CLR << 8 );
            CY_SET_REG32(PrISM_AUX_CONTROL2_PTR, PrISM_FIFO0_CLR | 
                                                            PrISM_FIFO0_CLR << 8 );
        #endif                                /* End PrISM_RESOLUTION */
        
        #if(PrISM_PULSE_TYPE_HARDCODED == 0)
            /* Writes density type provided by customizer */
            if(PrISM_GREATERTHAN_OR_EQUAL == 0 )
            {
                PrISM_CONTROL |= PrISM_CTRL_COMPARE_TYPE0_GREATER_THAN_OR_EQUAL;
            }
            else
            {
                PrISM_CONTROL &= ~PrISM_CTRL_COMPARE_TYPE0_GREATER_THAN_OR_EQUAL;
            }
    
            if(PrISM_GREATERTHAN_OR_EQUAL == 0)
            {
                PrISM_CONTROL |= PrISM_CTRL_COMPARE_TYPE1_GREATER_THAN_OR_EQUAL;
            }
            else
            {
                PrISM_CONTROL &= ~PrISM_CTRL_COMPARE_TYPE1_GREATER_THAN_OR_EQUAL;
            }
        #endif /* End PrISM_PULSE_TYPE_HARDCODED */
    }    /* End PrISM_initVar */
    #if(PrISM_PULSE_TYPE_HARDCODED == 0)
        /* Enable the PrISM computation */
        PrISM_CONTROL |= PrISM_CTRL_ENABLE;
    #endif /* End PrISM_PULSE_TYPE_HARDCODED */
}


/*******************************************************************************
* FUNCTION NAME:   PrISM_Stop
********************************************************************************
*
* Summary:
*  Stops PrISM computation. Outputs remain constant.
*
* Parameters:  
*  void
*
* Return: 
*  void
*
*******************************************************************************/
void PrISM_Stop(void)
{
    #if(PrISM_PULSE_TYPE_HARDCODED == 0)
        PrISM_CONTROL &= ~PrISM_CTRL_ENABLE;
    #else
        /* PulseTypeHardoded option enabled - to stop PrISM use enable input */
    #endif /* End $INSTANCE_NAME`_PULSE_TYPE_HARDCODED */
}

#if(PrISM_PULSE_TYPE_HARDCODED == 0)


    /***************************************************************************
    * FUNCTION NAME:     PrISM_SetPulse0Mode
    ****************************************************************************
    *
    * Summary:
    *  Sets the pulse density type for Density0. Less than or Equal(<=) or 
    *  Greater that or Equal(>=).
    *
    * Parameters:
    *  pulse0Type: Selected pulse density type.
    *
    * Return:
    *  void
    * 
    ***************************************************************************/
    void PrISM_SetPulse0Mode(uint8 pulse0Type)
    {
        if(pulse0Type == PrISM_GREATERTHAN_OR_EQUAL)
        {
            PrISM_CONTROL |= PrISM_CTRL_COMPARE_TYPE0_GREATER_THAN_OR_EQUAL;
        }
        else
        {
            PrISM_CONTROL &= ~PrISM_CTRL_COMPARE_TYPE0_GREATER_THAN_OR_EQUAL;
        }
    }
    
    
    /***************************************************************************
    * FUNCTION NAME:   PrISM_SetPulse1Mode
    ****************************************************************************
    *
    * Summary:
    *  Sets the pulse density type for Density1. Less than or Equal(<=) or 
    *  Greater that or Equal(>=).
    *
    * Parameters:  
    *  pulse1Type: Selected pulse density type.
    *
    * Return:
    *  void
    * 
    ***************************************************************************/
    void PrISM_SetPulse1Mode(uint8 pulse1Type)
    {
        if(pulse1Type == PrISM_GREATERTHAN_OR_EQUAL)
        {
            PrISM_CONTROL |= PrISM_CTRL_COMPARE_TYPE1_GREATER_THAN_OR_EQUAL;
        }
        else
        {
            PrISM_CONTROL &= ~PrISM_CTRL_COMPARE_TYPE1_GREATER_THAN_OR_EQUAL;
        }
    }

#endif /* End PrISM_PULSE_TYPE_HARDCODED == 0 */


/*******************************************************************************
* FUNCTION NAME:   PrISM_ReadSeed
********************************************************************************
*
* Summary:
*  Reads the PrISM Seed register.
*
* Parameters:
*  void
*
* Return:
*  Current Period register value.
*
*******************************************************************************/
uint16 PrISM_ReadSeed(void)
{
    return( CY_GET_REG16(PrISM_SEED_PTR) );
}


/*******************************************************************************
* FUNCTION NAME:   PrISM_WriteSeed
********************************************************************************
*
* Summary:
*  Writes the PrISM Seed register with the start value.
*
* Parameters:
*  seed: Seed register value.
*
* Return:
*  void
*
*******************************************************************************/
void PrISM_WriteSeed(uint16 seed)
{
    if(seed != 0)
    {
        CY_SET_REG16(PrISM_SEED_COPY_PTR, seed);
        CY_SET_REG16(PrISM_SEED_PTR, seed);
    }
}


/*******************************************************************************
* FUNCTION NAME:   PrISM_ReadPolynomial
********************************************************************************
*
* Summary:
*  Reads the PrISM polynomial.
*
* Parameters:
*  void
*
* Return:
*  PrISM polynomial.
*
*******************************************************************************/
uint16 PrISM_ReadPolynomial(void)
{
    return( CY_GET_REG16(PrISM_POLYNOM_PTR) );
}


/*******************************************************************************
* FUNCTION NAME:   PrISM_WritePolynomial
********************************************************************************
*
* Summary:
*  Writes the PrISM polynomial.
*
* Parameters:
*  polynomial: PrISM polynomial.
*
* Return:
*  void
*
*******************************************************************************/
void PrISM_WritePolynomial(uint16 polynomial)
{
    CY_SET_REG16(PrISM_POLYNOM_PTR, polynomial);
    
}


/*******************************************************************************
* FUNCTION NAME:   PrISM_ReadPusle0
********************************************************************************
*
* Summary:
*  Reads the PrISM Pulse Density0 register.
*
* Parameters:
*  void
*
* Return:
*  Pulse Density0 register value.
*
*******************************************************************************/
uint16 PrISM_ReadPusle0(void)
{
    return( CY_GET_REG16(PrISM_DENSITY0_PTR) );
}


/*******************************************************************************
* FUNCTION NAME:   PrISM_WritePulse0
********************************************************************************
*
* Summary:
*  Writes the PrISM Pulse Density0 register with the Pulse Density value.
*
* Parameters:
*  pulseDesity0: Pulse Density value.
*
* Return:
*  void
*
*******************************************************************************/
void PrISM_WritePulse0(uint16 pulseDesity0)
{
    CY_SET_REG16(PrISM_DENSITY0_PTR, pulseDesity0);
}


/*******************************************************************************
* FUNCTION NAME:   PrISM_ReadPusle1
********************************************************************************
*
* Summary:
*  Reads the PrISM Pulse Density1 register.
*
* Parameters:
*  void
*
* Return:
*  Pulse Density1 register value.
*
*******************************************************************************/
uint16 PrISM_ReadPusle1(void)
{
    return( CY_GET_REG16(PrISM_DENSITY1_PTR) );
}


/*******************************************************************************
* FUNCTION NAME:   PrISM_WritePulse1
********************************************************************************
*
* Summary:
*  Writes the PrISM Pulse Density1 register with the Pulse Density value.
*
* Parameters:
*  pulseDesity1: Pulse Density value.
*
* Return:
*  void
*
*******************************************************************************/
void PrISM_WritePulse1(uint16 pulseDesity1)
{
    CY_SET_REG16(PrISM_DENSITY1_PTR, pulseDesity1);
}


/* [] END OF FILE */
