//----------------------------------------------------------------------------
//                  Demo user interface software
//
//             "An Adaptive HF DSP Modem for 100 and 200 Baud"
//                              QEX, Nov 1994.		     
//									     
//                    (c) Johan Forrer, KC7WW				     
//		         26553 Priceview Drive                               
//                         Monroe, OR 97456				     
//----------------------------------------------------------------------------
//								       
//     Compilation/link instructions for Borland C++ 4.0                                  
//     >bcc -c -ml -O2 -Z -G demo.cpp
//     >tlink /x c:\bc4\lib\c0l demo PSA1.obj PSA2.obj, emu mathl cl
//								       
//----------------------------------------------------------------------------
//              This is how the AD1848 is reprogrammed:
//        Please the AD1848 data sheet for programming details.
//----------------------------------------------------------------------------
// Idx=0
// Sta=0
//  0 =0x00  select line with low gain
//  1 =0x00  slecet line with low gain
//  2 =0x80  mute left  aux#1
//  3 =0x80  mute right aux#1
//  4 =0x80  mute left  aux#2
//  5 =0x80  mute right aux#2
//  6 =0x07  left  DAC with some attenuation
//  7 =0x07  right DAC with some attenuation
//  8 =0x51  16 bit 2s compl, linear PCM, stereo, 
//           xtal1=16.93MHz, 5512.5 kHz,
//  9 =0x08  DMA capture/playback, Autocalibrate after mode change,
//           disable capture/playback, dual DMA mode
// 10 =0     N.A
// 11 =0     N.A
// 12 =0     N.A
// 13 =0     digital mix muted
// 14 =0     base count = 0
// 15 =0     base count = 0
//----------------------------------------------------------------------------

#include "psa.h"			// constants and includes 

#define DEFAULT_BGCOLOR	 LIGHTBLUE	// our custom screen colors
#define DEFAULT_CHCOLOR	 LIGHTGRAY
#define F1                  59		// simplify access to function keys
#define F2                  60
#define F3                  61
#define F4                  62
#define ALT_X               45

//----------------------------------------------------------------------------
// Local function prototypes
//----------------------------------------------------------------------------
void prepare_DSP(char *, unsigned char *);
void control_DSP(void);
void shutdown_DSP(void);
void run_user_interface();
void tuning_bars();
void pretty_print();
void Adjust_Sound_input_level(int);
void Adjust_Sound_output_level(int);
void set_modem_baudrate(int);
void set_AFSK_state(int);

#define MON "rte.ld" 		// define name of bootstrap
#define APP "modem.cde"		// define name of application

int peak_mark, peak_space;   	// Global data
int AF_source_in, Baudrate, Afsk;

//=====================START OF MAIN==========================================
void main()
{
unsigned int boot_version;
unsigned char initstate[18]={0,0,0x00,0x00,0x80,0x80,0x80,
		0x80,0x07,0x07,0x51,0x08,0,0,0,0,0,0};

// ---------------------------------------------------------------------------
// Check for presence and initialze DSP card
// ---------------------------------------------------------------------------
	if (!checkDSPcard(pssbase, wssbase)) {
		printf("No PSA card - or WSS address is not available\n");
		exit(0);
		}

//----------------------------------------------------------------------------
// Hard reset DSP
//----------------------------------------------------------------------------
	outpw(pssbase+PSS_CONTROL,PSS_RESET);	
	outpw(pssbase+PSS_CONTROL,0);	

//--------------------------------------------------------------------------
// Perform the DSP bootstrap next
// and verify that it has completed successfully
//--------------------------------------------------------------------------
	boot_version=bootdsp(MON, NULL, (unsigned)16000);
	checkdsptestprog();
	printf("Monitor version %d.%d reported\n", 
		boot_version>>8, boot_version&0x0F);

//----------------------------------------------------------------------------
// Now we are ready to load the DSP application
//----------------------------------------------------------------------------
	prepare_DSP(APP, initstate);		// Initialize CODEC

//----------------------------------------------------------------------------
// Prepare some initial sound levels. Note these may be a little
// different for a Cardinal card.
//	Source=0x00  - Line input for Orchid SW32
//	Source=0x40  - Line input for Cardinal
//	Source=0x80  - Mike input for Orchid
//----------------------------------------------------------------------------
	AF_source_in=0x80;		// I prefer to use Mike input
	Adjust_Sound_input_level(7);	// Nice levels for Orchid
	Adjust_Sound_output_level(35);
	Baudrate=100;			// 100 baud
	Afsk=0;				// AFSK is muted

//----------------------------------------------------------------------------
// Finally we are ready to run the DSP application
//----------------------------------------------------------------------------
	control_DSP();			        // Run user interface
	shutdown_DSP();				// Clean up after exit
}
//========================END OF MAIN========================================


//---------------------------------------------------------------------------
//                    Some local functions
//---------------------------------------------------------------------------

//-------------This is where the user interface goes-------------------------
void run_user_interface()
{
char ch;
	pretty_print();			// Print program's static text
	set_modem_baudrate(Baudrate);
	set_AFSK_state(Afsk);

	for(;;) {
		delay(40);		// slow down screen updates
		tuning_bars();		// update tuning displays

//----------------------------------------------------------------------------
// Service the user's keyboard
//----------------------------------------------------------------------------
		if(kbhit()) {
			ch=getch();
         		if(ch==0) {                	// 0 code
            			ch=getch();
            			switch(ch) {
               				case F1:        // F1
	                      			break;

               				case F2:        // F2
	                      			break;

               				case F3:        // F3
						if(Baudrate==100) 
							Baudrate=200;
						else    Baudrate=100;
						set_modem_baudrate(Baudrate);
	                      			break;

					case F4:	// F4
						if(Afsk==0) Afsk=1;
						else if(Afsk==1) Afsk=2;
						else if(Afsk==2) Afsk=0;
						set_AFSK_state(Afsk);
						break;
					case ALT_X:	 // Breakout
						ch=0x1B; // force ESC
						break;		
					}
				}
			if(ch==0x1B) break;
			}
		}
		
	while(kbhit()) getch();		// flush out breakout charater

//---------------------------------------------------------------------------
// Restore display to the state before we ran
//---------------------------------------------------------------------------
   	_setcursortype( _NORMALCURSOR );
	normvideo();
        window(1,1,80,25); clrscr();
}

//----------------------------------------------------------------------------
// Routine to load & run DSP application
//----------------------------------------------------------------------------
void prepare_DSP(char *app_name, unsigned char ad1848_state[])
{
	load_DSP_mem(app_name);	   // load the DSP application code

	set1848state(ad1848_state); // reprogram CODEC

	outp(wssbase+WSS_CODECINDEXADDR,0x0009); // Capture/Playback on
	outp(wssbase+WSS_CODECINDEXDATA,0x0003); 

	putdspword(0x00B4);        // send "start application" code to DSP 
}

//----------------------------------------------------------------------------
// Routine to terminate DSP application
//----------------------------------------------------------------------------
void shutdown_DSP()
{
	putdspword(0x00B5);	// send "stop application" code to DSP

	outp(wssbase+WSS_CODECINDEXDATA,0x0009); // Playback/capture off 
	outp(wssbase+WSS_CODECINDEXDATA, 0);

	outp(wssbase+WSS_CODECINDEXADDR,0x0006);  // mute DAC
	outp(wssbase+WSS_CODECINDEXDATA, 0xDF);
	outp(wssbase+WSS_CODECINDEXADDR,0x0007);
	outp(wssbase+WSS_CODECINDEXDATA, 0xDF);

}

//----------------------------------------------------------------------------
// Prepare and run the user's interface program
//
//      This is a four parter:
//
// 	1). Preparing the PC's interrupt system.
// 	2). Starting the DSP timer to allow low-level communication 
//          between the PC and DSP.
// 	3). Initiate and run the user's interface code until completion.
// 	4). Terminating the DSP timer and close down low-level
//          communication between the PC and DSP.
//----------------------------------------------------------------------------

void control_DSP()
//----------------------------------------------------------------------------
{
// Define ISR prototypes
void interrupt (*oldfunc)(...);		// for storage of old vector
void interrupt DSP_int_serv(...);  	// Prototype for new interrupt serv.
unsigned char old_IMR;			// Save old PIC IMR

//---------------------------------------------------------------------------
// Preparing the PC's Interrupt system
//---------------------------------------------------------------------------
	oldfunc=getvect(INT);		// Hook IRQ7 on the PC side
	setvect(INT,DSP_int_serv);

// Set INT7 on DSP side as well
	outpw(pssbase+PSS_CONFIG,IRQ7);
	old_IMR=inp(0x21);		// get IMR
	outp(0x21,old_IMR&0x7F);	// Arm PC's IRQ7 in IMR

//---------------------------------------------------------------------------
// Starting the DSP timer to allow low-level communication 
// between the PC and DSP.
//---------------------------------------------------------------------------
	putdspword(0x00B0);		// send "start timer" code to DSP

//---------------------------------------------------------------------------
// Initiate and run the user's interface code until completion.
//---------------------------------------------------------------------------
	run_user_interface();

//---------------------------------------------------------------------------
// Terminating the DSP timer and close down low-level
// communication between the PC and DSP.
//---------------------------------------------------------------------------
	putdspword(0x00B1);         	// send "stop timer" code to DSP

	outpw(pssbase+PSS_IRQ_ACK, 0);	// Clear any pending PSA interrupts
	outpw(pssbase+PSS_CONFIG,0x0000);  // Remove PSA from IRQ7

	outp(0x21,old_IMR);		// Restore IMR

	setvect(INT, oldfunc);		// Restore INT
}	

//----------------------------------------------------------------------------
// The new IRQ7 interrupt handler
//----------------------------------------------------------------------------
void interrupt DSP_int_serv(...)
{
#define DECAY 50			// Decay factor for tuning display
int mark, space, AD_data;

	outpw(pssbase+PSS_IRQ_ACK, 0);	// Clear the interrupt
	outp(0x20,0x20); 		// write EOI to PIC 
					// else bad things happen!

//------------------Process recovered data from DSP---------------------------
// Get signal from DSP demodulator. Separate into mark & space peak levels.			     
//----------------------------------------------------------------------------
	AD_data=inpw(pssbase+PSS_DATA);
	if(AD_data<0) {
		space=(-AD_data);
		if(space>peak_space) peak_space=space;
		else                 peak_space=peak_space-DECAY;
		}
	else	{
	      	mark =  AD_data;
		if(mark>peak_mark)   peak_mark=mark;
		else		     peak_mark=peak_mark-DECAY;
		}
}

//--------------------------------------------------------------------------
// Display tuning bars
//
// Channel data >=10 displays in RED
// Channel data =8 or 9 displays in YELLOW
//
// The raw channel data from the DSP is a single signed 16-bit value. It
// is first interpreted, i.e., positive values are MARK, negative
// values are SPACE. A simple peak-detector algorithm improves visualization.
// The peak values for MARK and SPACE are used in the display routine
// where they are scaled and clipped, and averaged to fit into the tuning 
// dispay template. 
//--------------------------------------------------------------------------
void tuning_bars()
{
#define TUN 32				// Center tuning display

int stat, i, j, mark_tuning, space_tuning;
static int mark, space;
static char upper_row[]="m¿m  ";   // Tuning display templates
static char lower_row[]="ss  ";

	mark_tuning=(peak_mark>>8);	// scale to a usable level
	space_tuning=(peak_space>>8);

	mark=(mark+mark_tuning)>>1;    // Average mark with prev.
	space=(space+space_tuning)>>1; // Average space with prev.

	if(mark>=10)  mark=10;
	if(space>=10) space=10;

	gotoxy(TUN,3);
	for(j=0; j<15; j++) cprintf("%c", upper_row[j]);
	gotoxy(TUN,4);
	for(j=0; j<15; j++) cprintf("%c", lower_row[j]);


	if(mark>=10) {
		textattr(0x7F&(RED+(DEFAULT_BGCOLOR<<4)));
		gotoxy(TUN+12-mark,3); 
		cprintf("");
		}
	else mark++;
	for(i=mark-1; i>=0 ;i--)
		if((mark>7)&&(i>=7)) {
			gotoxy(TUN+12-i,3); 
			textattr(0x7F&(YELLOW+(DEFAULT_BGCOLOR<<4)));
			cprintf("");
			}
		else 
			if((mark>4)&&(i>=4)) {
				gotoxy(TUN+12-i,3); 
				textattr(0x7F&(LIGHTBLUE+(DEFAULT_BGCOLOR<<4)));
				cprintf("");
				}
			else {
				gotoxy(TUN+12-i,3); 
				textattr(0x7F&(LIGHTGRAY+(DEFAULT_BGCOLOR<<4)));
				cprintf("");
				}

	if(space>=10) {
		textattr(0x7F&(RED+(DEFAULT_BGCOLOR<<4)));
		gotoxy(TUN+12-space,4); 
		cprintf("");
		}
	else space++;
	for(i=space-1; i>=0 ;i--)
		if((space>7)&&(i>=7)) {
			gotoxy(TUN+12-i,4); 
			textattr(0x7F&(YELLOW+(DEFAULT_BGCOLOR<<4)));
			cprintf("");
			}
		else
			if((space>4)&&(i>=4)) {
				gotoxy(TUN+12-i,4); 
				textattr(0x7F&(LIGHTBLUE+(DEFAULT_BGCOLOR<<4)));
				cprintf("");
				}
			else {
				gotoxy(TUN+12-i,4); 
				textattr(0x7F&(LIGHTGRAY+(DEFAULT_BGCOLOR<<4)));
				cprintf("");
				}

	textcolor(DEFAULT_CHCOLOR);
	textbackground(DEFAULT_BGCOLOR);
}

//----------------------------------------------------------------------------
// Static screen text
//----------------------------------------------------------------------------
void pretty_print()
{
	textbackground(DEFAULT_BGCOLOR); // Default screen colors
	textcolor(DEFAULT_CHCOLOR);
	_setcursortype(_NOCURSOR);
	clrscr();
	gotoxy(1,2); 
	cprintf("ͺ DEMO MODEM CONTROL (c) KC7WW");
	cprintf(" ");

	gotoxy(1,25); 
	textattr(0x7F&(YELLOW+(DEFAULT_BGCOLOR<<4)));
	cprintf("F1");
	textbackground(DEFAULT_BGCOLOR);
	textcolor(DEFAULT_CHCOLOR);
	cprintf("=Input level "); 
	textattr(0x7F&(YELLOW+(DEFAULT_BGCOLOR<<4)));
	cprintf("F2");
	textbackground(DEFAULT_BGCOLOR);
	textcolor(DEFAULT_CHCOLOR);
	cprintf("=Speaker level "); 
	textattr(0x7F&(YELLOW+(DEFAULT_BGCOLOR<<4)));
	cprintf("F3");
	textbackground(DEFAULT_BGCOLOR);
	textcolor(DEFAULT_CHCOLOR);
	cprintf("=Modem "); 
	textattr(0x7F&(YELLOW+(DEFAULT_BGCOLOR<<4)));
	cprintf("F4");
	textbackground(DEFAULT_BGCOLOR);
	textcolor(DEFAULT_CHCOLOR);
	cprintf("=AFSK ");
	textattr(0x7F&(YELLOW+(DEFAULT_BGCOLOR<<4)));
	cprintf("ALT-X or ESC");
	textbackground(DEFAULT_BGCOLOR);
	textcolor(DEFAULT_CHCOLOR);
	cprintf("=Quit");
	textbackground(DEFAULT_BGCOLOR);
	textcolor(DEFAULT_CHCOLOR);
}

//---------------------------------------------------------------------------
// This adjusts the level of the line in
//---------------------------------------------------------------------------
void Adjust_Sound_input_level(int level)
{
	asm {cli}
	putdspword(0x00BC);
	putdspword((AF_source_in+level)<<8);
	asm {sti}
}

//---------------------------------------------------------------------------
// This adjusts the level of the speaker level
//---------------------------------------------------------------------------
void Adjust_Sound_output_level(int level)
{
	asm {cli}
	putdspword(0x00BB);
	putdspword((0x007F-level)<<8);
	asm {sti}
}

//---------------------------------------------------------------------------
// This changes the modem's baudrate
//---------------------------------------------------------------------------
void set_modem_baudrate(int rate)
{
	asm {cli}
	if(rate==100) putdspword(0x00BE);
	else	      putdspword(0x01BE);
	asm {sti}

// Update display to reflect new baudrate
	gotoxy(60,3); cprintf("%d  - BAUD",rate);
}

//---------------------------------------------------------------------------
// This changes the AFSK generator's state
//---------------------------------------------------------------------------
void set_AFSK_state(int state)
{
	asm {cli}
	putdspword(0x00BD+(state<<8));
	asm {sti}

// Update display to reflect the new state
	gotoxy(60,4); 
	switch (state) {
		case 0: cprintf("AFSK - MUTED");
			break;
		case 1: cprintf("AFSK - MARK ");
			break;
		case 2: cprintf("AFSK - SPACE");
			break;
		}
}
