import serial
from time import sleep
import re

class PMSA(object):
	def __init__(self):
		self.p = serial.Serial()	# Default is 9600, 8 bits, no parity, 1 stop bit, COM1
		self.p.port = 0
		self.p.timeout = 5
		self.p.open()

	def getReply(self, cmd=None):
		if cmd:
			self.p.write(cmd)
		reply = ""
		while 1:
			ch = self.p.read(size=1)
			if ch == '\n' or ch == None:		# string terminator
				break
			reply += ch
		return  reply			# return as string

	def initSigGen(self):
		self.p.flushOutput()
		self.p.flushInput()
		init = ['*RST\n', 'SYST:COMM:SER:BAUD 9600\n', 'OUTP:STAT ON\n']
		for k in range(3):
			s = init[k]
			self.p.write('A')
			self.p.write(s)
			sleep(0.1)

	def setPowerLevel(self, dBm):
		self.p.write('A')
		pwr = 'POW:LEV:IMM:AMPL %f\n' % dBm
		self.p.write(pwr)

	def setFreq(self, f):
		self.p.write('A')
		nfd = 6		# number of fractional digits
		tmp = '%10.*f' % (nfd, f)
		nz = re.findall(r'0+\Z', tmp)	# match trailing zeros
		if nz:
			lnz = len(nz[0])
			if lnz == nfd:
				tmp = tmp[:-lnz+1]		# allow one trailing zero after the decimal point
			else:
				tmp = tmp[:-lnz]
		cmd = 'FREQ ' + tmp + ' MHz\n'
		self.p.write(cmd)
		# sleep() necessary for getReply to return, why ?
		# perhaps settling time for signal generator
		sleep(0.15)

	def scanFreq(self, startFreq, stopFreq, df, delaySec):
		N = int((stopFreq - startFreq)/df) + 1
		f = startFreq
		results = []
		for k in range(N):
			self.setFreq(f)
			response = self.getReply('B')
			results.append((f, response))
			sleep(delaySec)
			f += df
		return results

if __name__ == '__main__':
	def test6():
		ps = PMSA()
		f = 1.0
		ps.setFreq(f)
		for k in range(20):
			r = ps.getReply('B')
			print r[:-1]

	def test5():
		ps = PMSA()
		dbm = -20
		ps.setPowerLevel(dbm)

	def test4():
		ps = PMSA()
		case = 0
		if case == 0:
			fstart = 6.7
			fend = 7.1
			df = 0.1
		elif case == 1:
			fstart = int(1e6)
			fend = int(300e6)
			df = int(10e6)
		delaySec = 0.01
		reply = ps.scanFreq(fstart, fend, df, delaySec)
		for r in reply:
			print '%10s  %s' % (r[0], r[1][:-1])

	def test3():
		ps = PMSA()
		ps.p.write('n')
		reply = ps.getReply()	# should be "Now is the time"
		print len(reply), reply

	def test2():
		ps = PMSA()
		ps.initSigGen()

	def test1():
		pass

	def test0():
		ps = PMSA()
		ps.initSigGen()
		ps.setPowerLevel(-20)
		sleep(0.1)
		freq = 9.8
		d = []
		while 1:
			d.append(1)
			ps.setFreq(freq)
			d.append(2)
			r = ps.getReply('B')
			d.append(r[:-1])
			sleep(0.1)
			#break
		print d

	test = 4
	if test == 0:
		test0()
	elif test == 1:
		test1()
	elif test == 2:
		test2()			# initSigGen
	elif test == 3:
		test3()			# "Now is the time"
	elif test == 4:
		test4()			# scanFreq
	elif test == 5:
		test5()			# set power level
	elif test == 6:
		test6()			# setFreq

