from PMSA.pmsa import PMSA
from time import sleep, time
from math import sqrt, floor, ceil
from scipy import array, mean, empty
import matplotlib.pyplot as plt
from matplotlib.pyplot import figure, show, legend
import matplotlib.ticker as ticker
from scipy.stats import linregress
from pylab import subplots_adjust, xlim, ylim, setp
import shelve, os.path, datetime


class PMSA_PROCESS:
	def __init__(self, dirname):
		self.dbFilename = os.path.join(dirname, 'pmsaDatabase')
		self.calHistoryFilename = os.path.join(dirname, 'calHistory.txt')

	def calibrate(self, freq):
		"""
			Compute scale factor

			freq is the frequency at which the calibration is made.
		"""
		ps = PMSA()
		ps.initSigGen()
		ps.setFreq(freq)
		minDb = -90
		maxDb = 20
		dDb = 2.5
		N = int((maxDb - minDb)/dDb) + 1
		dBm = minDb
		resp = []
		pwr = []
		for k in range(N):
			ps.setPowerLevel(dBm)
			sleep(0.1)
			r = self.getResponse(ps, [freq])
			resp.extend(r)
			pwr.append(dBm)
			dBm += dDb
		ps.setPowerLevel(-30)

		regstart = 8	# - 70
		regend = N-2	# + 15
		slope, y0 = linregress(pwr[regstart:regend], resp[regstart:regend])[:2]
		sh = shelve.open(self.dbFilename)
		sh['slope'] = slope
		sh['Y0'] = y0
		sh.close()

		regLine = empty(N)
		for k in range(N):
			regLine[k] = y0 + slope*pwr[k]

		fig = figure()
		ax1 = fig.add_subplot(211)
		subplots_adjust(hspace=0.001)
		ax1.set_title('Response vs Signal Generator Output')
		xlim(-90, 20)
		ylim(0, 1023)
		ax1.plot(pwr, resp, marker='+')
		ax1.plot(pwr, regLine, ls=':')
		ax1.set_xlabel('dBm')
		ax1.set_ylabel('ADC Counts')
		handles, labels = ax1.get_legend_handles_labels()
		ax1.grid(True)
		xticklabels = ax1.get_xticklabels()
		setp(xticklabels, visible=False)
		x1 = pwr[regstart]
		x2 = pwr[regend-1]

		d = datetime.datetime.now()
		tmp = d.strftime('%m/%d/%Y %I:%M:%S %p')
		tmp += '  %6.1f  %6.3f  %6.3f  %2.0f  %2.0f\n' % (freq, slope, y0, x1, x2)
		try:
			fil = open(self.calHistoryFilename, 'a')
		except IOError:
			fil = open(self.calHistoryFilename, 'w')
		fil.write(tmp)
		fil.close()
		tmp = 'Slope: %6.3f Counts/dBm in range %2.0f to %2.0f dBm' % (slope, x1, x2)
		xcal = 0.2
		fig.text(xcal, 0.84, tmp)
		tmp = 'Y0:    %8.2f' % y0
		fig.text(xcal, 0.8, tmp)
		tmp = 'Frequency: %6.1f MHz' % freq
		fig.text(xcal, 0.76, tmp)

		dif = (resp - regLine) / slope
		ax2 = fig.add_subplot(212, sharex=ax1)
		ylim(-1, 1)
		ax2.plot(pwr, dif, marker='+')
		ax2.grid(True)
		ax2.set_ylabel('Conformance to regression line\ndBm                      ')
		ax2.set_xlabel('dB')
		show()

	def get3dBFrequencies(self, freqs, resp):

		def interpolate(y3db, k):
			"""
						 x1 - x2     x1*y2 - x2*y1
				x = y * -------- -  ---------------
						(y1 - y2)      (y1 - y2)

							   Fk - Fk+1    Fk*Yk+1 - Fk+1*Yk
				f3db = y3db * ----------- - -----------------
							  (Yk - Yk+1)     (Yk - Yk+1)
			"""
			yk = resp[k]
			yk1 = resp[k+1]
			fk = freqs[k]
			fk1 = freqs[k+1]
			f3db = (y3db*(fk - fk1) - (fk*yk1 - fk1*yk))/(yk - yk1)
			return f3db

		peak = max(resp)
		N = len(resp)
		p3dB = p1 = peak - 3.0
		# Scan response values from the left
		for k in range(N):
			p2 = resp[k]
			if p2 > p1:
				# p2 is the first value greater than the -3dB value
				# Interpolate to find the frequency of the -3dB value
				p0 = resp[k-1]
				f3dbL = interpolate(p1, k-1)
				k3dbL = k-1
				break

		# Scan from the right
		for k in range(N):
			j = N - k - 1
			p2 = resp[j]
			if p2 > p1:
				p0 = resp[j+1]
				f3dbR = interpolate(p1, j+1)
				k3dbR = j
				break
		return p3dB, f3dbL, f3dbR

	def getResponse(self, ps, freqs, delay=0):
		results = []
		for f in freqs:
			ps.setFreq(f)
			r = ps.getReply('B')
			results.append(int(r))
		return array(results)

	def plotFreqResponse(self, freqs, dBm, atten, yL, yH, plotTitle, \
							mode=1, findBandwidth=False):
		"""
			freq	-	List of frequencies in MHz
			dBm		-	Signal generator power level
			yL		-	Plot lower limit in dB
			yH		-	Plot upper limit in dB
			atten	-	Value added to plot, dB
			mode	-	X-axis: 0-semilog, 1-linear
			findBandwidth -	If true response bandwidth is indicated on plot
		"""
		sh = shelve.open(self.dbFilename, flag='r')
		slope = sh['slope']
		Y0 = sh['Y0']
		sh.close()

		ps = PMSA()
		ps.initSigGen()
		ps.setPowerLevel(dBm)
		sleep(0.1)
		resp = self.getResponse(ps, freqs)
		ps.setPowerLevel(-135)	# default

		fig = figure()
		ax1 = fig.add_subplot(111)
		ax1.set_title(plotTitle)
		ax1.grid(True)
		if yL and yH:
			ylim(yL, yH)
		resp_dB = (resp - Y0)/slope	# resp is an np array
		resp_dB += atten
		if mode == 0:
			ax1.semilogx(freqs, resp_dB, marker='+')
		else:
			ax1.plot(freqs, resp_dB, marker='+')
		ax1.set_xlabel('Frequency - MHz')
		ax1.set_ylabel('Response - dB')
		formatter = ticker.FormatStrFormatter('%3.2f')
		ax1.xaxis.set_major_formatter(formatter)

		N = len(resp_dB)
		if 0:
			for k in range(N):
				print '%3d  %6.2f   %6.2f' % (k, freqs[k], resp_dB[k])
		if findBandwidth:
			p3dB, f3dBL, f3dBR = self.get3dBFrequencies(freqs, resp_dB)
			print p3dB, f3dBL, f3dBR
			ax1.scatter([f3dBL, f3dBR], [p3dB, p3dB], marker='+', lw=5)
			tmp = 'Bandwidth: %6.3f MHz' % (f3dBR - f3dBL)
			fig.text(0.2, 0.84, tmp)
		show()

if __name__ == '__main__':
	# Frequencies have units MHz
	freqs  = [0.1*k for k in range(1, 10)]
	freqs += [k for k in range(1, 10)]
	freqs += [10*k for k in range(1, 10)]
	freqs += [100*k for k in range(1, 5)]
	freqs = array(freqs)

	dirname = 'C:\Documents and Settings\Owner\My Documents\Python\PMSA'
	dbFilename = os.path.join(dirname, 'errDatabase')
	pmsa = PMSA_PROCESS(dirname)

	case = 0
	if case == 0:
		sigGen = -30	# dBm
		corr = 0
		yH = 0 #sigGen+1
		yL = sigGen-2
		title = 'Response vs Frequency'
		atten = 0
		pmsa.plotFreqResponse(freqs, sigGen, atten, yL, yH, title, mode=0)
	elif case == 1:
		freq = 10.0		# MHz
		pmsa.calibrate(freq)
