/*
 * fpp.c
 *
 * Flash PIC programmer.  Download a hex file to a flash PIC
 * using the "erase before programming" serial programming algorithm.
 * The software doesn't care what it tries to program!
 *
 * Revision history:
 *
 * 16-Oct-2001: V-0.0; created based on PP.C V-0.5
 * 17-Oct-2001: V-0.1; added -f option
 * 20-Oct-2001: V-0.2; added predefined device list
 * 25-Oct-2001: V-0.3; renamed some things
 * 28-Oct-2001: V-0.4; added Windows messages and -m option
 * 28-Oct-2001: V-0.5; added -u option
 * 28-Oct-2001: V-0.6; use Windows to open hex file
 * 30-Oct-2001: V-0.7; can now use Windows GUI
 * 01-Nov-2001: V-0.8; moved some functions to mess.c
 * 15-Nov-2001: V-0.9; hardware can be defined on the command line
 * 20-Feb-2002: V-0.9; corrected PIC sizes for build 42.
 *
 * Copyright (C) 2002 David Tait.  All rights reserved.
 * Permission is granted to use, modify, or redistribute this software
 * so long as it is not sold or exploited for profit.
 *
 * THIS SOFTWARE IS PROVIDED AS IS AND WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "fpp.h"

#define LDCONF  0               /* serial mode commands */
#define LDPROG  2
#define RDPROG  4
#define INCADD  6
#define BEGPRG  8
#define LDDATA  3
#define RDDATA  5
#define ERPROG  9
#define ERDATA  11

DEVICE device[] = {             /* list of known devices and sizes */
    {"16F83", 83, 512, 64}, {"16F84", 84, 1024, 64},
    {"16F85", 85, 1024, 64}, {"16F86", 86, 2048, 64},
    {"16F627", 627, 1024, 128}, {"16F628", 628, 2048, 128},
    {"16F870", 870, 2048, 64}, {"16F871", 871, 2048, 64},
    {"16F872", 872, 2048, 64}, {"16F873", 873, 4096, 128},
    {"16F874", 874, 4096, 128}, {"16F876", 876,  8192, 256},
    {"16F877", 877, 8192, 256}, {NULL, 0, MAXPM, MAXDM}
};

int psize = MAXPM;              /* program memory size in words */
int dsize = MAXDM;              /* data memory size in bytes */
int debug = 0;                  /* true if debug requested */
int usewin = 0;                 /* use Windows GUI if true */
int mplab = 0;                  /* allow Windows messages if true */
int silent = 0;                 /* no text output (mplab overrides) */
int got_file = 0;               /* true if filename given in command line */
int erase = 0;                  /* erase PIC if true */
int mode = STOP;                /* let PIC run mode = GO */
int setcfg = 0;                 /* true if config set by command-line */
int setid = 0;                  /* true if ID words set by GUI */
int no_read = 0;                /* do not read from PIC if true */
int pause = 0;                  /* pause before quitting */
char *part;                     /* points to part number (if any) */
static U16 clconfig;            /* config from command line (if any) */
static int dump = 0;            /* dump PIC to a file if true */
static int verify_only = 0;     /* verify PIC against a file if true */
static int update = 0;          /* read PIC before programming */
static int purge = 0;           /* if true the registry is to be purged */
static int store = 0;           /* if true store current setup in registry */

static char *pname = PNAME;
static char *version = VERSION;
static char *does = DOES;
static char *copyright = COPYRIGHT " " AUTHOR ".";
static char *email = EMAIL;

static void inithw(void);
static void usage_err(char *s);
static void get_option(char *s);
static void load(char *fn);
static void dump_all(FILE *fp);
static int program_all(void);
static int verify_all(void);
static int read_all(void);
static U16 get_conf(void);


int fpp(int argc, char *argv[])
{
    FILE *fp;
    int i, c, t;
    char *fn = NULL;
    time_t start_t;

    erasehex(MAXPM, MAXDM, 14);   /* initialise buffers */

    reg_open();
    reg_getall();                 /* recover registry values */

    for ( i=1; i<argc; ++i ) {
        if ( (c = *argv[i]) == '-' || c == '/' )
            get_option(++argv[i]);
        else {
            if ( got_file )
                usage_err("Only one file can be specified");
            fn = argv[i];
            got_file = 1;
        }
    }

    if ( purge ) {                /* purge registry and quit */
        reg_purge();
        quit(NULL,SUCCESS);
    }

    if ( store ) {                /* store command line settings and quit */
        reg_putall();
        quit(NULL,SUCCESS);
    }

    if ( !usewin || mplab || debug )
        get_console();

    if ( mplab && !debug )
        pause = 0;

    if ( debug )                  /* ignore all other switches */
        inithw();                 /* will never return */

    inithw();                     /* check H/W and set valid appropriately */

#ifdef __WIN32__
    if ( !valid || usewin ) {     /* ignore most switches */
        usage();
        info("\n\n    Waiting for a Windows application to complete ...\n\n");
        load(fn);
        reg_putall();             /* update registry just in case ... */
        fppgui();
        info("\n\n    Completed ...\n\n");
        quit(NULL,SUCCESS);
    }
#else /* !__WIN32__ */
    if ( !valid )
        usage_err("No hardware specified");
    if ( usewin )
        usage_err("Not running Windows");
#endif

    if ( silent && !mplab )       /* no interaction if silent */
         wait = 0;

    if ( (dump || verify_only) && !got_file )
        usage_err("Need a file");


    if ( verify_only && !can_read )
        usage_err("Hardware cannot verify device");

    if ( !erase && !(mode==GO)
                           && !setcfg && !got_file )  /* file needed */
        usage_err("Need a file");

    if ( wait && (erase || (setcfg && !got_file)) )
        if ( !query("OK to program/read?", 0) )  /* ask user */
            quit("Cancelled",FAIL);

    if ( dump ) {
        if ( (fp = fopen(fn,"r")) != NULL ) {  /* file exists? */
            fclose(fp);
            if (silent)
                quit("Dump would overwrite existing file",FAIL);
            if ( !query("Overwrite existing file?", 1) )
                quit("Cancelled",FAIL);
       }
       if ( read_all() ) {
           if ( (fp = fopen(fn,"w")) == NULL )
               quit("Cannot create hexfile",FAIL);
           dump_all(fp);
           fclose(fp);
           quit(NULL,SUCCESS);
       }
       quit("PIC is erased - nothing to dump",FAIL);
    }

    if (!got_file && !setcfg && mode == GO )
        quit(NULL,SUCCESS);

    load(fn);

    start_t = time(NULL);
    if ( got_file && verify_only ) {
        if ( !verify_all() )
            quit("Verify failed",FAIL);
    } else {
        if ( erase ) {
            erase_all();
            if ( !got_file && !setcfg )
                quit(NULL,SUCCESS);
        }
        if ( got_file || setcfg ) {
            if ( !program_all() )
                quit("Verify failed during programming",FAIL);
            if ( can_read )
                message("Config: %04X\n", get_conf());
        }
    }
    t = (int) (time(NULL)-start_t);
    message("Finished in %d sec%c\n\n",t,(t!=1)? 's': ' ');
    quit(NULL,SUCCESS);
    return 0;                  /* never reached */
}



static void inithw(void)
{
    message("%s  %s  %s\n\n",does,version,copyright);
    if ( setup() == -1 )
        quit("Check setup",FAIL);
}


void usage(void)
{
   info("Usage: %s  [ -devgqsnumwcrphft! ]  hexfile\n\n", pname);
   info("        d = dump,   e = erase,  v = verify,  g = go\n"
        "        q = debug,  s = silent, n = no read, ! = no wait\n"
        "        u = update, m = mplab,  wF = winui,  cHHHH = config\n"
        "        r = record, p = purge,  hS:D = hardware\n"
        "        fDDDD:DDDD = device,    tDD:DD = timing\n\n");
   info("Bug reports to %s\n\n",email);
}


static void usage_err(char *s)
{
   if ( !silent )
       usage();
   quit(s, FAIL);
}


static void fuses(char *sp)          /* extract hex config */
{
    int n;

    clconfig = 0;
    while ( (n = hex(*sp)) >= 0 ) {
        clconfig = clconfig*16 + n;
        ++sp;
    }
}

static char *chop(char *sp)          /* chop string at ':' character */
{
    while ( *sp ) {
        if ( *sp == ':' ) {
            *sp = '\0';
            return ++sp;
        }
        ++sp;
    }
    return sp;
}


static void size(char *s1)           /* extract device size from arg */
{
    char *s2 = chop(s1);
    int n, set = 0;

    if ( strlen(s1) > 0 ) {
        psize = atoi(s1);
        set = 1;
    }
    if ( strlen(s2) > 0 )
        dsize = atoi(s2);
    else if ( set ) {                /* only one size given? */
        for ( n=0; device[n].part && device[n].pn != psize; ++n )
            ;
        if ( device[n].part ) {        /* it could be a device number */
            part = device[n].part;
            psize = device[n].pm;
            dsize = device[n].dm;
        }
    }
    if ( psize > MAXPM )
        psize = MAXPM;
    if ( dsize > MAXDM )
        dsize = MAXDM;
}

static void timing(char *s1)         /* get delay values from arg */
{
    char *s2 = chop(s1);

    if ( strlen(s1) > 0 )
        prgdly = atoi(s1);
    if ( strlen(s2) > 0 )
        pwrdly = atoi(s2);
}

static void hardware(char *s1)       /* get hardware descriptor from arg */
{
    char *s2 = chop(s1);

    if ( strlen(s1) > 0 ) {
        hw.hws = s1;
        hw.hwname = "User defined";
        hw.hwid = -1;
    }
    if ( strlen(s2) > 0 )
        hwport = atoi(s2);
}


static void get_option(char *s)
{
   char *sp;

   for ( sp=s; *sp; ++sp ) {
      switch ( *sp ) {
         case 'q':
         case 'Q': debug = 1; pause = 1; break;
         case 'e':
         case 'E': erase = 1; break;
         case 'd':
         case 'D': dump = 1; break;
         case 'v':
         case 'V': verify_only = 1; break;
         case 'g':
         case 'G': mode = GO; break;
         case 's':
         case 'S': silent = 1; break;
         case 'm':
         case 'M': mplab = 1; break;
         case 'n':
         case 'N': no_read = 1; break;
         case 'f':
         case 'F': size(++sp); sp = " "; break;  /* ignore rest of arg */
         case 'h':
         case 'H': hardware(++sp); sp = " "; break;
         case 't':
         case 'T': timing(++sp); sp = " "; break;
         case 'p':
         case 'P': purge = 1; break;
         case 'r':
         case 'R': store = 1; break;
         case 'w':
         case 'W': usewin = (*(++sp) == '0')? 0: 1;
                   if (!usewin) pause = 1;
                   break;
         case 'u':
         case 'U': update = 1; break;
         case 'c':
         case 'C': fuses(++sp); setcfg = 1; sp = " "; break;
         case '!': wait = 0; break;
         case '-':
         case '/': break;
         default: usage_err("Unknown switch");
      }
   }
}


static void load(char *fn)
{
    FILE *fp;
    int e;

    if ( got_file ) {
        if ( update ) {
            if ( wait && !query("OK to program/read?", 0) )
                quit("Aborted",FAIL);
            read_all();
        }
        message("Opening %s ...\n", fn);
        if ( (fp = fopen(fn,"r")) == NULL )
            quit("Cannot open hexfile",FAIL);
        if ( (e = load_all(fp)) < 0 )
            quit(errhex(e),FAIL);
    }
    if ( setcfg )
        config = clconfig;           /* use command line config */
}


static void prog_cycle(U16 w)
{
    out_word(w);
    command(BEGPRG);
    ms_delay(prgdly);
}


static void load_conf(int base)
{
    int i, n;

    command(LDCONF);
    out_word(0x3FFF);

    n = base - IBASE;
    for ( i=0; i<n; ++i )
        command(INCADD);
}


static void bulk_erase(void)
{
   command(1);
   command(7);
   command(BEGPRG);
   ms_delay(prgdly);
   command(1);
   command(7);
}


int load_all(FILE *fp)
{
    return loadhex(fp, psize, dsize, IBASE, CBASE, DBASE);
}


static void dump_all(FILE *fp)
{
    format = dumpfmt;
    dumphex(fp, pmlast+1, dmlast-DBASE+1, id, cf, DBASE);
}


void save_all(FILE *fp)
{
    int ibase = setid? IBASE: id;
    int cbase = setcfg? CBASE: cf;

    format = dumpfmt;
    dumphex(fp, pmlast+1, dmlast-DBASE+1, ibase, cbase, DBASE);
}


void erase_code(void)
{
    message("Erasing code only ...\n");

    prog_mode();                 /* this _might_ work */
    command(LDPROG);
    out_word(0x3FFF);
    command(ERPROG);
    command(BEGPRG);
    ms_delay(prgdly);

    if ( usewin )
       message("Done\n");
}


void erase_data(void)
{
    message("Erasing data only ...\n");

    prog_mode();                 /* this _might_ work */
    command(LDDATA);
    out_word(0x3FFF);
    command(ERDATA);
    command(BEGPRG);
    ms_delay(prgdly);

    if ( usewin )
       message("Done\n");
}


void erase_all(void)
{
    message("Erasing all ...\n");

    prog_mode();                 /* defeat code protection */
    load_conf(CBASE);            /* should erase entire device, but ... */
    bulk_erase();

    prog_mode();
    command(LDPROG);             /* bulk erase program memory */
    out_word(0x3FFF);
    bulk_erase();

    prog_mode();
    command(LDDATA);             /* bulk erase data memory */
    out_word(0x3FFF);
    bulk_erase();

    if ( usewin )
       message("Done\n");
}


static int program(U16 *buf, int n, U16 mask, int ldcmd, int rdcmd, int base)
{
    int i;
    U16 r, w;

    prog_mode();

    if ( base >= IBASE && base <= CBASE )
        load_conf(base);

    for ( i=0; i<n; ++i ) {
        w = buf[i]&mask;
        if ( can_read ) {
            command(rdcmd);
            r = in_word() & mask;
            if ( w != r ) {
                message(mplab?"":"%04X\r",i);
                command(ldcmd);
                prog_cycle(w);
                command(rdcmd);
                r = in_word() & mask;
                if ( w != r ) {
                    message("%s: %04X: read %04X, wanted %04X\n",
                                                 pname,base+i,r,w);
                    return 0;
                }
            }
        } else {
            message(mplab?"":"%04X\r",i);
            command(ldcmd);
            prog_cycle(w);
        }
        command(INCADD);
    }
    return 1;
}


int program_code(void)
{
    message("Programming ...\n");
    if ( pmlast >= 0 && !program(progbuf,pmlast+1,0x3FFF,LDPROG,RDPROG,0) )
        return 0;
    message("Done\n");

    return 1;
}

int program_data(void)
{
    message("Programming data ...\n");
    if ( dmlast >= 0 &&
                !program(databuf,dmlast-DBASE+1,0xFF,LDDATA,RDDATA,DBASE) )
        return 0;
    message("Done\n");

    return 1;
}

int program_ID(void)
{
    message("Programming ID ...\n");
    if ( (id > 0 || setid) && !program(idbuf,4,0x3FFF,LDPROG,RDPROG,IBASE) )
        return 0;
    message("Done\n");

    return 1;
}

int program_conf(void)
{
    message("Programming config ...\n");
    if ( (cf > 0 || setcfg) &&
                           !program(&config,1,0x3FFF,LDPROG,RDPROG,CBASE) )
        return 0;
    message("Done\n");

    return 1;
}


static int program_all(void)
{
    message("Programming ...\n");

    if ( pmlast >= 0 && !program(progbuf,pmlast+1,0x3FFF,LDPROG,RDPROG,0) )
        return 0;
    if ( dmlast >= 0 &&
                !program(databuf,dmlast-DBASE+1,0xFF,LDDATA,RDDATA,DBASE) )
        return 0;
    if ( ((id > 0 && psize != 0) || setid) &&
                             !program(idbuf,4,0x3FFF,LDPROG,RDPROG,IBASE) )
        return 0;
    if ( ((cf > 0 && psize != 0) || setcfg) &&
                           !program(&config,1,0x3FFF,LDPROG,RDPROG,CBASE) )
        return 0;

    return 1;
}


static U16 get_conf(void)
{
    int i;

    prog_mode();
    command(LDCONF);
    out_word(0x3FFF);
    for ( i=0; i<7; ++i )
        command(INCADD);
    command(RDPROG);
    return (in_word() & 0x3FFF);
}


static int verify(U16 *buf, int n, U16 mask, int rdcmd, int base)
{
    int i;
    U16 r, w;

    prog_mode();

    if ( base >= IBASE && base <= CBASE )
        load_conf(base);

    for ( i=0; i<n; ++i ) {
        message(mplab?"":"%04X\r",i);
        command(rdcmd);
        r = in_word() & mask;
        if ( (w = buf[i]&mask) != r ) {
            message("%s: %04X: read %04X, wanted %04X\n",
                                                pname,base+i,r,w);
            return 0;
        }
        command(INCADD);
    }
    return 1;
}


int verify_code(void)
{
    message("Verifying ...\n");
    if ( pmlast >= 0 && !verify(progbuf,pmlast+1,0x3FFF,RDPROG,0) )
        return 0;
    message("Done\n");

    return 1;
}

int verify_data(void)
{
    message("Verifying data ...\n");
    if ( dmlast >= 0 && !verify(databuf,dmlast-DBASE+1,0xFF,RDDATA,DBASE) )
        return 0;
    message("Done\n");

    return 1;
}

int verify_ID(void)
{
    message("Verifying ID ...\n");
    if ( (id > 0 || setid ) && !verify(idbuf,4,0x3FFF,RDPROG,IBASE) )
        return 0;
    message("Done\n");

    return 1;
}

int verify_conf(void)
{
    message("Verifying config ...\n");
    if ( (cf > 0 || setcfg) && !verify(&config,1,0x3FFF,RDPROG,CBASE) )
        return 0;
    message("Done\n");

    return 1;
}


static int verify_all(void)
{
    message("Verifying ...\n");

    if ( pmlast >= 0 && !verify(progbuf,pmlast+1,0x3FFF,RDPROG,0) )
        return 0;
    if ( dmlast >= 0 && !verify(databuf,dmlast-DBASE+1,0xFF,RDDATA,DBASE) )
        return 0;
    if ( ((id > 0 && psize != 0) || setid) &&
                             !verify(idbuf,4,0x3FFF,RDPROG,IBASE) )
        return 0;
    if ( ((cf > 0 && psize != 0) || setcfg) &&
                           !verify(&config,1,0x3FFF,RDPROG,CBASE) )
        return 0;

    return 1;
}


int read_code(void)
{
    int i;

    message("Reading ...\n");
    pmlast = -1;
    prog_mode();
    for ( i=0; i<psize; ++i ) {
        command(RDPROG);
        if ( (progbuf[i] = in_word() & 0x3FFF) != 0x3FFF )
            pmlast = i;
        command(INCADD);
    }
    message("Done\n");

    return pmlast != -1;
}

int read_data(void)
{
    int i;

    message("Reading data ...\n");
    dmlast = -1;
    prog_mode();
    for ( i=0; i<dsize; ++i ) {
        command(RDDATA);
        if ( (databuf[i] = in_word() & 0xFF) != 0xFF )
            dmlast = i;
        command(INCADD);
    }
    if ( dmlast >= 0 )
        dmlast += DBASE;
    message("Done\n");

    return dmlast != -1;
}

int read_ID(void)
{
    int i;

    message("Reading ID ...\n");
    id = 0;
    prog_mode();
    command(LDCONF);
    out_word(0x3FFF);
    for ( i=0; i<4; ++i ) {
        command(RDPROG);
        if ( (idbuf[i] = in_word() & 0x3FFF) != 0x3FFF )
            id = IBASE;
        command(INCADD);
    }
    message("Done\n");

    return id != 0;
}

int read_conf(void)
{
    message("Reading config ...\n");
    cf = 0;
    if ( (config = get_conf()) != 0x3FFF )
        cf = CBASE;
    message("Done\n");

    return cf != 0;
}


static int read_all(void)
{
    int i;

    message("Reading ...\n");

    pmlast = -1;
    dmlast = -1;
    id = 0;
    cf = 0;

    if ( psize != 0 ) {
        prog_mode();
        for ( i=0; i<psize; ++i ) {
            command(RDPROG);
            if ( (progbuf[i] = in_word() & 0x3FFF) != 0x3FFF )
                pmlast = i;
            command(INCADD);
        }
    }
    if ( dsize != 0 ) {
        prog_mode();
        for ( i=0; i<dsize; ++i ) {
            command(RDDATA);
            if ( (databuf[i] = in_word() & 0xFF) != 0xFF )
                dmlast = i;
            command(INCADD);
        }
        if ( dmlast >= 0 )
            dmlast += DBASE;        /* changed from pp V-0.5 */
    }
    if ( psize != 0 ) {
        prog_mode();
        command(LDCONF);
        out_word(0x3FFF);
        for ( i=0; i<4; ++i ) {
            command(RDPROG);
            if ( (idbuf[i] = in_word() & 0x3FFF) != 0x3FFF )
                id = IBASE;
            command(INCADD);
        }
        for ( i=0; i<3; ++i )
            command(INCADD);
        command(RDPROG);
        if ( (config = in_word() & 0x3FFF) != 0x3FFF )
            cf = CBASE;
    }

    return !(pmlast == -1 && dmlast == -1 && id == 0 && cf == 0 );
}

