/*
 * fppgui.c
 *
 * GUI for Flash PIC programmer.  NOT Unicode friendly.
 *
 * Revision history:
 *
 * 29-Oct-2001: V-0.0; created
 * 01-Nov-2001: V-0.1; added Setup dialog
 * 16-Nov-2001; V-0.2; moved Setup stuff to setup.c
 * 17-Jan-2002; V-0.3; removed View buffer size check (failed in XP).
 *
 * Copyright (C) 2002 David Tait.  All rights reserved.
 * Permission is granted to use, modify, or redistribute this software
 * so long as it is not sold or exploited for profit.
 *
 * THIS SOFTWARE IS PROVIDED AS IS AND WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED.
 *
 */

#include "fpp.h"
#ifdef __WIN32__
#include <windows.h>
#include <commctrl.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <tchar.h>
#include "resource.h"


#define MAX_TEXT   60000   /* more than enough space to dump an 8K PIC */
#define NVWORDS    4       /* number of hex words to view per line */
#define NCHARS     16      /* number of ASCII chars to view per line */
#define FWIDTH     8       /* Font size for View Window */
#define FHEIGHT    12
#define VIEWFONT   ANSI_FIXED_FONT
#define BLANK     TEXT("\r\n    --- Blank ---")

#define SUCCESS    0       /* already defined in fpp.h */
#define FAIL       1       /* '' */
#define WAIT       2
#define DONE       3

#define NIBBLE(w,i)  (((w)&(0xF<<(4*(i))))>>(4*(i)))

extern HINSTANCE hInst;    /* in main.c */
HWND hDlg;


enum control {
    LOAD=0, SAVE, RELOAD, INIT, READ, PROGRAM, VERIFY, ERASE, RUN,
    ABOUT, SETUP, QUIT, STATUS, VIEW, CSIZE, DSIZE, VIEWCODE,
    VIEWDATA, VIEWHEX, VIEWASCII, CVAL, IDVAL, PCODE, PDATA,
    PID, PCONFIG, NCONTROLS
};

static int iCon[] = {
    IDC_LOAD, IDC_SAVE, IDC_RELOAD, IDC_INIT, IDC_READ, IDC_PROGRAM,
    IDC_VERIFY, IDC_ERASE, IDC_RUN, IDC_ABOUT, IDC_SETUP, IDC_QUIT,
    IDC_STATUS, IDC_VIEW, IDC_PSIZE, IDC_DSIZE, IDC_VIEWCODE,
    IDC_VIEWDATA, IDC_VIEWHEX, IDC_VIEWASCII, IDC_CVAL, IDC_IDVAL,
    IDC_PCODE, IDC_PDATA, IDC_PID, IDC_PCONFIG
};

static HWND hCon[NCONTROLS];

static char ViewBuf[MAX_TEXT+1];
static char *pBuf = ViewBuf;
static BOOL running = FALSE;

/* lists of known device sizes plus spaces for user defined sizes */

static char *devps[] = {
    "512", "1024", "2048", "4096", "8192", NULL, NULL};
static char *devds[] = {
    "64", "128", "256", NULL, NULL};


BOOL CALLBACK DlgSetup(HWND h, UINT msg, WPARAM wParam, LPARAM lParam);


/* SetGuiName()
 *
 * Display program name and part number in caption.
 *
 */

void SetGuiName(void)
{
    sprintf(ViewBuf, part[0]? "%s - [%s]": "%s", PNAME, part);
    SetWindowText(hDlg, ViewBuf);
}

/* Status()
 *
 * Display a status message.
 *
 */

static void Status(TCHAR *s, int type)
{
    HCURSOR hWait;
    static HCURSOR hOld = NULL;

    if ( hOld )
        SetCursor(hOld);
    hOld = NULL;

    switch ( type ) {
    case WAIT:
        hWait = LoadCursor(NULL, IDC_WAIT);
        hOld = SetCursor(hWait);
        break;

    case FAIL:
        MessageBeep(MB_ICONEXCLAMATION);
        break;

    case DONE:
        MessageBeep(MB_OK);
        break;
    }
    SetDlgItemText(hDlg, IDC_STATUS, s);
}


/* CheckBox()
 *
 * Checks/unchecks a variable number of checkboxes.
 *
 */

static void CheckBox(const int n, const BOOL tick, ...)
{
    int i;
    va_list argp;
    WPARAM p = (WPARAM) BST_CHECKED;

    va_start(argp, tick);

    if ( tick == FALSE )
        p = (WPARAM) BST_UNCHECKED;

    for (i=0; i<n; ++i)
        SendMessage(hCon[va_arg(argp,int)], BM_SETCHECK, p, 0);

    va_end(argp);
}


/* Checked()
 *
 * Returns TRUE if the given checkbox is checked
 *
 */


static BOOL Checked(enum control i)
{
    if ( IsDlgButtonChecked(hDlg, iCon[i]) == BST_CHECKED )
        return TRUE;
    return FALSE;
}


/* HexLine(), HexView(), AsciiLine(), AsciiView()
 *
 * Functions used to format the PIC buffer for display.
 *
 */


static void HexLine(int a, int n)
{
    int i, w = 4;
    U16 *Mem = progbuf;

    sprintf(pBuf, "%04X: ", a);
    pBuf += 6;
    if ( a >= DBASE ) {
        Mem = databuf;
        a -= DBASE;
        w = 2;
    }
    for (i=0; i<n-1; ++i) {
        sprintf(pBuf, w==4?"%04X ":"%02X ", Mem[a+i]);
        pBuf += w + 1;
    }
    sprintf(pBuf, w==4?"%04X\r\n":"%02X\r\n", Mem[a+i]);
    pBuf += w + 2;
}

static void HexView(int base)
{
    int a, n = pmlast + 1, wpl = NVWORDS;

    if ( base == DBASE ) {
        n = dmlast - DBASE + 1;
        wpl *= 2;
    }

    if ( n > 0 ) {
        pBuf = ViewBuf;
        for (a=0; a < n - wpl; a += wpl)
            HexLine(base + a, wpl);
        if ( a < n )
            HexLine(base + a, n - a);
        *pBuf = '\0';
    }
    else
    sprintf(ViewBuf, BLANK);
    SetDlgItemText(hDlg, IDC_VIEW, ViewBuf);
}

static void AsciiLine(int a, int n)
{
    int i, c;
    U16 *Mem = progbuf;

    sprintf(pBuf, "%04X: ", a);
    pBuf += 6;
    if ( a >= DBASE ) {
        Mem = databuf;
        a -= DBASE;
    }
    for (i=0; i<n; ++i)
        *pBuf++ = ((c = Mem[a+i]&0xFF) >= ' ' && c <= 0x7F )? c: '.';
    *pBuf++ = '\r';
    *pBuf++ = '\n';
}

static void AsciiView(int base)
{
    int a, n = pmlast + 1;

    if ( base == DBASE )
        n = dmlast - DBASE + 1;

    if ( n > 0 ) {
        pBuf = ViewBuf;
        for (a=0; a < n - NCHARS; a += NCHARS)
            AsciiLine(base + a, NCHARS);
        if ( a < n )
            AsciiLine(base + a, n - a);
        *pBuf = '\0';
    }
    else
    sprintf(ViewBuf, BLANK);
    SetDlgItemText(hDlg, IDC_VIEW, ViewBuf);
}


/* ReadHex()
 *
 * Read a hex number from a control.  Returns -1 if
 * the value is not set.
 *
 */

static int ReadHex(enum control i)
{
    char *sp = ViewBuf;
    int n, val = 0;

    GetDlgItemText(hDlg, iCon[i], ViewBuf, 10);

    if ( strcmp(ViewBuf, "####") == 0 )
        return -1;

    while ( (n = hex(*sp)) >= 0 ) {
        val = val*16 + n;
        ++sp;
    }
    return val;
}


/* CheckHex()
 *
 * Checks for valid hex input ("not set" - #### - is valid).
 *
 */

static void CheckHex(enum control i, int max)
{
    char *sp = ViewBuf;
    int n, pos, val = 0;

    GetDlgItemText(hDlg, iCon[i], ViewBuf, 10);

    if ( strcmp(ViewBuf, "####") == 0 )
        return;

    while ( (n = hex(*sp)) >= 0 ) {
        val = val*16 + n;
        ++sp;
    }

    if ( *sp ) {         /* non-hex character found */
        MessageBeep(MB_OK);
        *sp = 0;
        pos = sp - ViewBuf;
        SetDlgItemText(hDlg, iCon[i], ViewBuf);
        SendMessage(hCon[i], EM_SETSEL, pos, pos);
    }
    if ( val > max ) {       /* exceeded max */
        MessageBeep(MB_OK);
        sprintf(ViewBuf, "%04X", max);
        SetDlgItemText(hDlg, iCon[i], ViewBuf);
        SendMessage(hCon[i], EM_SETSEL, 4, 4);
    }

    return;
}

/* SetVals()
 *
 * Called before most actions.  Stops the PIC if running and
 * sets  Config and ID words from controls.
 *
 */

static void SetVals(int stop)
{
    int i, w;

    if ( stop ) {
        SetDlgItemText(hDlg, IDC_RUN, TEXT("Ru&n"));
        run_mode(STOP);
        running = FALSE;
    }
    if ( (w = ReadHex(IDVAL)) >= 0 ) {
        for ( i=0; i<4; ++i )
            idbuf[i] = NIBBLE(w,(3-i));
        setid = 1;
    }
    else
    setid = 0;
    if ( (w = ReadHex(CVAL)) >= 0 ) {
        config = w;
        setcfg = 1;
    }
    else
    setcfg = 0;

    i = SendMessage(hCon[CSIZE], CB_GETCURSEL, 0, 0);
    psize = atoi(devps[i]);
    if ( pmlast >= psize ) {
        pmlast = psize - 1;
        Status(TEXT("Warning: some code lost"), FAIL);
        for ( i=pmlast+1; i<MAXPM; ++i )
            progbuf[i] = 0x3FFF;
    }

    i = SendMessage(hCon[DSIZE], CB_GETCURSEL, 0, 0);
    dsize = atoi(devds[i]);
    if ( dmlast-DBASE >= dsize ) {
        dmlast = dsize + DBASE - 1;
        Status(TEXT("Warning: some data lost"), FAIL);
        for ( i=dmlast-DBASE+1; i<MAXDM; ++i )
            databuf[i] = 0xFF;
    }
}


/* ShowConfig()
 *
 * Updates the config display.
 *
 */

static void ShowConfig(void)
{
    sprintf(ViewBuf, (cf||setcfg)?"%04X":"####", config);
    SetDlgItemText(hDlg, IDC_CVAL, ViewBuf);
}


/* ShowID()
 *
 * Updates the ID display.
 *
 */

static void ShowID(void)
{
    int i, idlocs = 0;

    for (i=0; i<4; ++i)
        idlocs = (idlocs<<4) + (idbuf[i]&0xF);

    sprintf(ViewBuf, (id||setid)? "%04X":"####", idlocs);
    SetDlgItemText(hDlg, IDC_IDVAL, ViewBuf);
}


/* ShowView()
 *
 * Updates the buffer View, ID and config displays.
 *
 */

static void ShowView(int base, int view)
{
    CheckBox(2, TRUE, (base == DBASE)? VIEWDATA:VIEWCODE,
        (view == VIEWHEX)? VIEWHEX:VIEWASCII );
    CheckBox(2, FALSE, (base == DBASE)? VIEWCODE:VIEWDATA,
        (view == VIEWHEX)? VIEWASCII: VIEWHEX );
    (view == VIEWHEX)? HexView(base): AsciiView(base);
    ShowConfig();
    ShowID();
}


/* InitSize()
 *
 * Initialises a size combobox.
 *
 */

static void InitSize(enum control control)
{
    static char ps[20], ds[20];    /* hold user defined sizes */
    char **ip, **p, *s;
    int i, size;

    if (control == CSIZE) {
        s = ps;
        p = devps;
        size = psize;
    }
    else {
        s = ds;
        p = devds;
        size = dsize;
    }

    i = -1;
    ip = p;
    while ( *ip ) {
        if ( atoi(*ip) == size )
            i = ip - p;
        SendMessage(hCon[control], CB_ADDSTRING, 0, (LPARAM) *ip++);
    }
    if ( i >= 0 )
        SendMessage(hCon[control], CB_SETCURSEL, i, 0);
    else {               /* user defined program size */
        sprintf(s, "%d", size);
        *ip = s;
        SendMessage(hCon[control], CB_ADDSTRING, 0, (LPARAM) *ip);
        SendMessage(hCon[control], CB_SETCURSEL, ip-p, 0);
    }
}

static void SetSize(enum control control)
{
    char **ip, **p;
    int i, size;

    if (control == CSIZE) {
        p = devps;
        size = psize;
    }
    else {
        p = devds;
        size = dsize;
    }

    i = -1;
    ip = p;
    while ( *ip )
        if ( atoi(*ip++) == size )
            i = ip - p - 1;
    if ( i >= 0 )
        SendMessage(hCon[control], CB_SETCURSEL, i, 0);
}



static TCHAR *SetupInfo(void)
{
    static TCHAR s[100];
    TCHAR *sp = s;

    sp += sprintf(sp, "H/W: %s on ", hw.hwname);

    if ( hwport == 0 )
        sprintf(sp, "NULL");
    else switch (hw.type) {
    case SERIAL:
        sprintf(sp, "COM%d", hwport);
        break;
    case PARALLEL:
        sprintf(sp, "LPT%d", hwport);
        break;
    default:
        sprintf(sp, "NULL");
    }
    return s;
}


/* DlgAbout
 *
 * Program info Dialog
 *
 */

static BOOL CALLBACK DlgAbout(HWND h, UINT msg, WPARAM wParam, LPARAM lParam)
{
    switch (msg) {

    case WM_INITDIALOG:


        sprintf(ViewBuf, "\r\n" PNAME "\r\n"
            DOES "\r\n"
            VERSION "\r\n\r\n"
            COPYRIGHT "\r\n"
            AUTHOR "\r\n\r\n%s", SetupInfo());

        SetDlgItemText(h, IDC_APANE, ViewBuf);
        return TRUE;

    case WM_COMMAND:
        switch (LOWORD(wParam)) {
        case IDOK:
            EndDialog(h, 1);
            return TRUE;
        }
        break;

    case WM_CLOSE:
        EndDialog(h, 0);
        return TRUE;

    }
    return FALSE;
}

static void EnableControls(void)
{
    int i;

    if ( valid )
        for (i=0; i<NCONTROLS; ++i)
            EnableWindow(hCon[i], TRUE);
    else {
        for (i=0; i<NCONTROLS; ++i)
            EnableWindow(hCon[i], FALSE);
        EnableWindow(hCon[QUIT], TRUE);
        EnableWindow(hCon[ABOUT], TRUE);
        EnableWindow(hCon[SETUP], TRUE);
        SetFocus(hCon[SETUP]);
    }
    if ( !can_read ) {
        EnableWindow(hCon[READ], FALSE);
        EnableWindow(hCon[VERIFY], FALSE);
    }
    if ( !can_run )
        EnableWindow(hCon[RUN], FALSE);
}


/* FppDlg()
 *
 * The GUI itself.
 *
 */

static BOOL CALLBACK FppDlg(HWND h, UINT msg, WPARAM wParam, LPARAM lParam)
{
    static FILE *fp;                 /* use stdio */
    static TCHAR fn[MAX_PATH+1], bfn[MAX_PATH+1];
    static OPENFILENAME ofn;
    static HFONT hFont;
    static int base = 0, view = VIEWHEX;
    LOGFONT lf;
    int i;

    switch (msg) {

    case WM_INITDIALOG:
        hDlg = h;
#ifdef __GNUC__
        SetClassLong(h, GCL_HICON, (LONG)
            LoadIcon(hInst, MAKEINTRESOURCE(IDC_FPPICON)));
#endif
        SetGuiName();
        ofn.hwndOwner = h;
        ofn.lStructSize = sizeof(OPENFILENAME);
        ofn.lpstrFile = fn;
        ofn.nMaxFile = MAX_PATH;
        ofn.lpstrFileTitle = bfn;
        ofn.nMaxFileTitle = MAX_PATH;
        ofn.lpstrFilter = "Hex files (*.hex)\0*.hex\0"
        "All files (*.*)\0*.*\0\0";
        ofn.lpstrDefExt = "hex";
        ofn.Flags = OFN_OVERWRITEPROMPT | OFN_HIDEREADONLY;
        fn[0] = '\0';
        for (i=0; i<NCONTROLS; ++i)
            hCon[i] = GetDlgItem(h, iCon[i]);
        SendMessage(hCon[VIEW], EM_SETLIMITTEXT, 0, 0);  /* 64K buffer */
        i = SendMessage(hCon[VIEW], EM_GETLIMITTEXT, 0, 0);
/*        if ( i < MAX_TEXT ) {
            MessageBeep(MB_ICONEXCLAMATION);
            EndDialog(h, 0);
        }
*/
        hFont = GetStockObject(VIEWFONT);
        GetObject(hFont, sizeof(LOGFONT), &lf);
        lf.lfWidth = FHEIGHT;
        lf.lfHeight = FWIDTH;
        hFont = CreateFontIndirect(&lf);
        SendMessage(hCon[VIEW], WM_SETFONT, (WPARAM) hFont, 0);
        Status(TEXT(DOES), 0);
        ShowView(base, view);
        CheckBox(4, TRUE, PCODE, PDATA, PID, PCONFIG);
        InitSize(CSIZE);
        InitSize(DSIZE);
        SendMessage(hCon[IDVAL], EM_LIMITTEXT, 4, 0);
        SendMessage(hCon[CVAL], EM_LIMITTEXT, 4, 0);
        ShowWindow(h, SW_SHOWNORMAL);    /* first call ignored */
        ShowWindow(h, SW_SHOWNORMAL);    /* forces window to be shown */
        EnableControls();
        return TRUE;

    case WM_CTLCOLORSTATIC:
        if ( (HWND) lParam == hCon[VIEW] ||
            (HWND) lParam == hCon[STATUS] )
            return (BOOL) GetStockObject(WHITE_BRUSH);
        else
        return FALSE;


    case WM_COMMAND:

        switch (LOWORD(wParam)) {

        case IDC_LOAD:
            Status(TEXT("Loading ..."), WAIT);
            SetVals(0);
            if ( GetOpenFileName(&ofn) ) {
                if ( (fp = fopen(fn,"r")) == NULL ) {
                    Status(TEXT("Can't open hexfile!"), FAIL);
                    return TRUE;
                }
                if ( (i = load_all(fp)) < 0 )
                    Status(TEXT("Error in hexfile!"), FAIL);
                else {
                    sprintf(ViewBuf, "Loaded %s", bfn);
                    if ( i > 0 )
                        sprintf(ViewBuf,
                            "Warning: some records ignored");
                    Status(ViewBuf, (i>0)? FAIL: 0);
                    setid = 0;
                    setcfg = 0;
                    if ( pmlast < 0 )
                        base = DBASE;
                    if ( dmlast < 0 )
                        base = 0;
                    ShowView(base, view);
                }
                fclose(fp);
            }
            else
            Status(TEXT("Cancelled"), 0);
            return TRUE;

        case IDC_RELOAD:
            if ( fn[0] == '\0' ) {
                SendMessage(h, WM_COMMAND, (WPARAM) IDC_LOAD, 0);
                return TRUE;
            }
            Status(TEXT("Reloading ..."), WAIT);
            SetVals(0);
            if ( (fp = fopen(fn,"r")) == NULL ) {
                Status(TEXT("Can't open hexfile!"), FAIL);
                return TRUE;
            }
            if ( (i = load_all(fp)) < 0 )
                Status(TEXT("Error in hexfile!"), FAIL);
            else {
                sprintf(ViewBuf, "Reloaded %s", bfn);
                if ( i > 0 )
                    sprintf(ViewBuf,
                        "Warning: some records ignored");
                Status(ViewBuf, (i>0)? FAIL: 0);
                setid = 0;
                setcfg = 0;
                if ( pmlast < 0 )
                    base = DBASE;
                if ( dmlast < 0 )
                    base = 0;
                ShowView(base, view);
            }
            fclose(fp);
            return TRUE;

        case IDC_SAVE:
            Status(TEXT("Saving ..."), 0);
            SetVals(0);
            if (pmlast==-1 && dmlast==-1 && id==0 && cf==0
                && !setid && !setcfg) {
                Status(TEXT("Nothing to save!"), FAIL);
                return TRUE;
            }
            if ( GetSaveFileName(&ofn) ) {
                if ( (fp = fopen(fn,"w")) == NULL ) {
                    Status(TEXT("Can't open hexfile!"), FAIL);
                    return TRUE;
                }
                else {
                    save_all(fp);
                    sprintf(ViewBuf, "Saved %s", bfn);
                    Status(ViewBuf, 0);
                    fclose(fp);
                }
            }
            else
            Status(TEXT("Cancelled"), 0);
            return TRUE;

        case IDC_INIT:
            Status(TEXT("Initialising ..."), WAIT);
            SetVals(0);
            for (i=0; i<psize; ++i)
                progbuf[i] = 0x3FFF;
            for (i=0; i<dsize; ++i)
                databuf[i] = 0xFF;
            pmlast = -1;
            dmlast = -1;
            id = 0;
            cf = 0;
            setid = 0;
            setcfg = 0;
            ShowView(base, view);
            Status(TEXT("Initialised!"), 0);
            return TRUE;

        case IDC_ERASE:
            Status(TEXT("Erasing ..."), WAIT);
            SetVals(1);
            if ( !Checked(PCODE) && !Checked(PDATA)
                && !Checked(PCONFIG) ) {
                if ( Checked(PID) )
                    Status("Cannot erase ID alone", FAIL);
                else
                Status("Nothing to erase", 0);
                return TRUE;
            }
            if ( wait && !query("Establish programming mode ...\r\n"
                "OK to erase?", 0) ) {
                Status(TEXT("Cancelled"), 0);
                return TRUE;
            }
            if ( !Checked(PCONFIG) ) {
                if ( Checked(PCODE) )
                    erase_code();
                if ( Checked(PDATA) )
                    erase_data();
                Status("Erased!", DONE);
            }
            else {
                if (!query("Erasing config erases all\r\nOK?", 0)) {
                    Status(TEXT("Cancelled"), 0);
                    return TRUE;
                }
                erase_all();
                Status(TEXT("Erased all!"), DONE);
            }
            run_mode(STOP);
            break;

        case IDC_READ:
            Status(TEXT("Reading ..."), WAIT);
            SetVals(1);
            if ( wait && !query("Establish programming mode ...\r\n"
                "OK to read?", 0) ) {
                Status(TEXT("Cancelled"), 0);
                return TRUE;
            }
            if ( Checked(PCODE) ) {
                read_code();
                if ( !Checked(PDATA) && base == DBASE )
                    base = 0;
            }
            if ( Checked(PDATA) ) {
                read_data();
                if ( !Checked(PCODE) && base == 0 )
                    base = DBASE;
            }
            if ( Checked(PID) ) {
                setid = 0;
                read_ID();
            }
            if ( Checked(PCONFIG) ) {
                setcfg = 0;
                read_conf();
            }
            Status(TEXT("Read!"), DONE);
            ShowView(base, view);
            run_mode(STOP);
            return TRUE;

        case IDC_PROGRAM:
            Status(TEXT("Programming ..."), WAIT);
            SetVals(1);
            if ( wait && !query("Establish programming mode ...\r\n"
                "OK to program?", 0) ) {
                Status(TEXT("Cancelled"), 0);
                return TRUE;
            }
            if ( Checked(PCODE) && !program_code() )
                Status(TEXT("Failed to program code!"), FAIL);
            else if ( Checked(PDATA) && !program_data() )
                Status(TEXT("Failed to program data!"), FAIL);
            else if ( Checked(PID) && !program_ID() )
                Status(TEXT("Failed to program ID!"), FAIL);
            else if ( Checked(PCONFIG) && !program_conf() )
                Status(TEXT("Failed to program config!"), FAIL);
            else
            Status(TEXT("Success!"), DONE);
            run_mode(STOP);
            return TRUE;

        case IDC_VERIFY:
            Status(TEXT("Verifying ..."), WAIT);
            SetVals(1);
            if ( wait && !query("Establish programming mode ...\r\n"
                "OK to verify?", 0) ) {
                Status(TEXT("Cancelled"), 0);
                return TRUE;
            }
            if ( Checked(PCODE) && !verify_code() )
                Status(TEXT("Failed to verify code!"), FAIL);
            else if ( Checked(PDATA) && !verify_data() )
                Status(TEXT("Failed to verify data!"), FAIL);
            else if ( Checked(PID) && !verify_ID() )
                Status(TEXT("Failed to verify ID!"), FAIL);
            else if ( Checked(PCONFIG) && !verify_conf() )
                Status(TEXT("Failed to verify config!"), FAIL);
            else
            Status(TEXT("Verified!"), DONE);
            run_mode(STOP);
            return TRUE;

        case IDC_RUN:
            if ( running ) {
                run_mode(STOP);
                SetDlgItemText(hDlg, IDC_RUN, TEXT("Ru&n"));
                running = FALSE;
                return TRUE;
            }
/*            if ( wait && !query("OK to run?", 0) ) {
                Status(TEXT("Cancelled"), 0);
                return TRUE;
            }
*/
            run_mode(GO);
            SetDlgItemText(hDlg, IDC_RUN, TEXT("S&top"));
            running = TRUE;
            return TRUE;

        case IDC_SETUP:
            Status(TEXT("Setup ..."), WAIT);
            SetVals(1);
            DialogBox(hInst, MAKEINTRESOURCE(IDD_SETUP),
                h, (DLGPROC) DlgSetup);
            if ( setup() != 0 )
                Status(TEXT("Setup error"), FAIL);
            SetSize(CSIZE);
            SetSize(DSIZE);
            SetGuiName();
            EnableControls();
            Status(SetupInfo(), 0);
            return TRUE;

        case IDC_ABOUT:
            DialogBox(hInst, MAKEINTRESOURCE(IDD_ABOUT),
                h, (DLGPROC) DlgAbout);
            return TRUE;

        case IDC_VIEWCODE:
            ShowView(base = 0, view);
            return TRUE;

        case IDC_VIEWDATA:
            ShowView(base = DBASE, view);
            return TRUE;

        case IDC_VIEWHEX:
            ShowView(base, view = VIEWHEX);
            return TRUE;

        case IDC_VIEWASCII:
            ShowView(base, view = VIEWASCII);
            return TRUE;

        case IDC_QUIT:
            SetVals(1);
            DeleteObject(hFont);
            EndDialog(h, 0);
            return TRUE;

        case IDC_IDVAL:
            if ( HIWORD(wParam) == EN_UPDATE )
                CheckHex(IDVAL, 0xFFFF);
            if ( HIWORD(wParam) == EN_KILLFOCUS ) {
                SetVals(0);
                ShowID();
            }
            return TRUE;

        case IDC_CVAL:
            if ( HIWORD(wParam) == EN_UPDATE )
                CheckHex(CVAL, 0x3FFF);
            if ( HIWORD(wParam) == EN_KILLFOCUS ) {
                SetVals(0);
                ShowConfig();
            }
            return TRUE;

        case IDC_PSIZE:
            if ( HIWORD(wParam) == CBN_SELCHANGE ) {
                part = "";
                SetGuiName();
                SetVals(0);
                ShowView(base, view);
            }
            return TRUE;

        case IDC_DSIZE:
            if ( HIWORD(wParam) == CBN_SELCHANGE ) {
                part = "";
                SetGuiName();
                SetVals(0);
                ShowView(base, view);
            }
            return TRUE;

        }
        break;

    case WM_CLOSE:
        DeleteObject(hFont);
        EndDialog(h, 0);
        return TRUE;

    }
    return FALSE;
}


int fppgui(void)
{
    return DialogBox(hInst, MAKEINTRESOURCE(IDD_GUI), NULL, (DLGPROC) FppDlg);
}

#endif /* __WIN32__ */

