/*
 * hw.c
 *
 * Hardware level routines for flash PIC programmer.
 * The info on PC interfacing was picked up from various places
 * over the years.  Nowadays everything can be found on
 * Craig Peacock's pages at http://www.beyondlogic.org
 *
 * Revision history:
 *
 * 21-Oct-2001: V-0.0; created
 * 22-Oct-2001: V-0.1; added debug
 * 23-Oct-2001: V-0.2; can now use the serial port
 * 02-Nov-2001: V-0.3; added some test modes
 * 15-Nov-2001: V-0.4; setup() split into several separate functions
 * 01-Oct-2002: V-0.5; fixed PARPIC setup 
 *
 * Copyright (C) 2001 David Tait.  All rights reserved.
 * Permission is granted to use, modify, or redistribute this software
 * so long as it is not sold or exploited for profit.
 *
 * THIS SOFTWARE IS PROVIDED AS IS AND WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED.
 *
 */

#include "fpp.h"
#ifdef __WIN32__
#include <windows.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


       /*  define outportb and inportb (how depends on compiler) */

#ifdef _MSC_VER
#define outportb(p,v)        _outp(p,v)
#define inportb(p)           _inp(p)
#endif

#ifdef __LCC__
#undef outportb
#define outportb(p,v)        _outp(p,v)
#undef inportb
#define inportb(p)           _inp(p)
#endif

#ifdef __GNUC__
unsigned char inportb(unsigned short port);
void outportb(unsigned short port, unsigned char c);
#endif

#define DATA    0
#define STATUS  1
#define CONTROL 2
#define MCR     3
#define MSR     4
#define LCR     5
#define BIT0    (1<<4)
#define BIT1    ((1<<1)<<4)
#define BIT2    ((1<<2)<<4)
#define BIT3    ((1<<3)<<4)
#define BIT4    ((1<<4)<<4)
#define BIT5    ((1<<5)<<4)
#define BIT6    ((1<<6)<<4)
#define BIT7    ((1<<7)<<4)
#define INV     0x1000
#define REGMASK 0xF
#define BITMASK 0xFF0
#define NREGS    6             /* number of I/O regs */
#define NPPINS   17            /* number of active parallel port pins */
#define DEFID    8             /* default hardware is TOPIC */


static char *version = "hw.c V-0.4  Copyright (C) 2001 David Tait";

#ifdef __WIN32__
static HANDLE hPort = NULL;
#endif

HW hw;                         /* struct describing hardware */
int hwport = -1;               /* hardware port number (-1,0,1,2,3,4) */
int prgdly = -1;               /* programming cycle delay */
int pwrdly = -1;               /* VDD to PGM and PGM to MCLR delay */
int tdly = -1;                 /* tiny (I/O) delay */
int wait = -1;                 /* prog mode asserted manually */
int can_read = -1;             /* whether hardware can read PIC */
int can_run = -1;              /* whether the hardware supports GO mode */
int valid = 0;                 /* validity stamp issued by setup() */

int dumpfmt = INHX8M;

static struct {
    int adr;                   /* I/O register address */
    int val;                   /* register contents */
} io[NREGS];

static int out, clk, in, vdd;  /* the register and bit position of each  */
static int mclr, pgm, read;    /* programmer signal */


HWSETUP known[] = {
    { 0,  "PPA-OCVM----I-------",  "7406/4066" },
    { 1,  "PPA-ocvm----i-------",  "7407/4066" },
    { 2,  "PPA-OCvm----I-------",  "7406/PNP" },
    { 3,  "PPA-ocVM----i-------",  "7407/PNP" },
    { 4,  "PIM-oc--------------",  "QandD (LPT)" },
    { 5,  "SIM---o--c--",          "QandD (COM)" },
    { 6,  "PIM-ocM-----I-------",  "PARPIC" },
    { 7,  "SIM--MO--CI-",          "SERPIC" },
    { 8,  "PIAcM--------i--o-Pr",  "TOPIC" },
    { 9,  "PPA-OCVM----I-------",  "Maplin" },
    { 10, "PPA-ocmp----i-------",  "TLVP" },
    { 11, "SPM--po--ci-",          "COMPORT" },
    { 12, "PIAcrv-------i--o-mp",  "TOPIC2" },
    { 13, "SIM--Mo--ci-",          "PMICP" },
    { 14, NULL,  NULL }          /* keep this! */
};

int bases[] = { 0,                            /* I/O base addresses */
                0x378, 0x3BC, 0x278, 0x378,
                0x3F8, 0x2F8, 0x3E8, 0x2E8
              };


char *hwversion(void)
{
    return version;
}

void assertp(int doit)
{
    if ( hwport <= 0 )
        return;

    if ( doit ) {
        if ( hw.type == SERIAL ) {
            outportb(io[MCR].adr, io[MCR].val);
            outportb(io[LCR].adr, io[LCR].val);
        } else {
            outportb(io[DATA].adr, io[DATA].val);
            outportb(io[CONTROL].adr, io[CONTROL].val^0xB);
        }
    }
}


static unsigned inbit(int iobit)
{
    int bit = (iobit&BITMASK)>>4;
    int reg = iobit&REGMASK;
    int invert = iobit&INV;
    int in_bits;

    if ( hwport <= 0 || bit == 0 || (hw.type == SERIAL && reg != MSR)
                  || (hw.type == PARALLEL && reg != STATUS) )
        return 1;

    in_bits = (reg == STATUS)?
                  inportb(io[STATUS].adr)^0x80: inportb(io[MSR].adr);
    return (invert)? ~in_bits&bit: in_bits&bit;
}



static void setbit(int iobit)
{
    int bit = (iobit&BITMASK)>>4;
    int reg = iobit&REGMASK;
    int invert = iobit&INV;

    if ( hwport <= 0 || bit == 0 || reg == STATUS || reg == MSR )
        return;

    if ( invert )
        io[reg].val &= ~bit;
    else
        io[reg].val |= bit;
}


static void clrbit(int iobit)
{
    int bit = (iobit&BITMASK)>>4;
    int reg = iobit&REGMASK;
    int invert = iobit&INV;

    if ( hwport <= 0 || bit == 0 || reg == STATUS || reg == MSR )
        return;

    if ( invert )
        io[reg].val |= bit;
    else
        io[reg].val &= ~bit;
}


unsigned inval(void)
{
    return inbit(in);
}

void outhi(int doit)
{
    setbit(out);
    assertp(doit);
}

void outlo(int doit)
{
    clrbit(out);
    assertp(doit);
}

void clkhi(int doit)
{
    setbit(clk);
    assertp(doit);
}

void clklo(int doit)
{
    clrbit(clk);
    assertp(doit);
}

void vddhi(int doit)
{
    setbit(vdd);
    assertp(doit);
}

void vddlo(int doit)
{
    clrbit(vdd);
    assertp(doit);
}

void mclrhi(int doit)
{
    setbit(mclr);
    assertp(doit);
}

void mclrlo(int doit)
{
    clrbit(mclr);
    assertp(doit);
}

void pgmhi(int doit)
{
    setbit(pgm);
    assertp(doit);
}

void pgmlo(int doit)
{
    clrbit(pgm);
    assertp(doit);
}

void readlo(int doit)
{
    clrbit(read);
    assertp(doit);
}

void readhi(int doit)
{
    setbit(read);
    assertp(doit);
}


void prog_mode(void)
{
    vddhi(0), readlo(0), clklo(0), outlo(0), mclrlo(1);
    us_delay(tdly);
    pgmlo(1);
    ms_delay(pwrdly);
    pgmhi(1);
    us_delay(tdly);
    mclrhi(1);
}


void run_mode(int mode)
{
    if ( can_run && mode == GO ) {
        mclrlo(1);
        us_delay(tdly);
        pgmlo(1);
        us_delay(tdly);
        if ( !usewin && wait )
            query("OK to run?", 0);
        readlo(0), clkhi(0), outhi(0), mclrhi(1);  /*  PIC should now run */
    } else {
        readlo(0), clklo(0), outlo(0), mclrlo(1);
        us_delay(tdly);
        pgmlo(1);
        ms_delay(pwrdly);
        vddlo(1);
    }
}


static void clock_out(int bit)
{
    bit? outhi(0): outlo(0); clkhi(1);
    us_delay(tdly);
    clklo(1);
    us_delay(tdly);
    outlo(1);
}

    /* out_word(w)
     *
     * Write a 14-bit value to the PIC.
     *
     */

void out_word(int w)
{
    int b;

    clock_out(0);
    for ( b=0; b<14; ++b )
        clock_out(w&(1<<b));
    clock_out(0);
}


static int clock_in(void)
{
    clkhi(1);
    us_delay(tdly);
    clklo(1);
    us_delay(tdly);
    return inbit(in)? 1: 0;
}

    /* in_word()
     *
     * Obtain a 14-bit value from the PIC.
     *
     */

int in_word(void)
{
    int b, w;

    readhi(0), outhi(1);
    (void) clock_in();
    for ( w=0, b=0; b<14; ++b )
        w += clock_in()<<b;
    (void) clock_in();
    readlo(1);

    return w;
}

    /* command(cmd)
     *
     * Send a 6-bit command to the PIC.
     *
     */

void command(int cmd)
{
    int b;

    outlo(1);
    us_delay(tdly);
    for ( b=0; b<6; ++b )
        clock_out(cmd&(1<<b));
    outhi(1);
    us_delay(tdly);
}


void cleanup(void)
{
#ifdef __WIN32__
    if ( hPort )
        CloseHandle(hPort);
    reg_putall();                    /* update registry */
    reg_close();
#endif
}


    /* giveio()
     *
     * Attempt to open the giveio driver.  Returns 0 if not opened.
     * Failure can be ignored if not running Win2k/NT.
     *
     */

static int giveio(void)
{
#ifdef __WIN32__
    HANDLE h = CreateFile("\\\\.\\giveio", GENERIC_READ, 0, NULL,
                                OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);

    if(h != INVALID_HANDLE_VALUE) {
         CloseHandle(h);
         return 1;
    }
#endif
    return 0;
}


static void showpin(char *function, int pin)
{
    int k = (hw.type == SERIAL)? NPPINS: 0;

    static char *pinname[] = {
        "", "/STROBE (1)", "D0 (2)", "D1 (3)", "D2 (4)", "D3 (5)",
        "D4 (6)", "D5 (7)", "D6 (8)", "D7 (9)", "/ACK (10)",
        "BUSY (11)", "PAPEREND (12)", "SELECT (13)", "/AUTOLF (14)",
        "/ERROR (15)", "/INIT (16)", "/SELECTIN (17)",
        "DCD (1)", "RD (2)", "TD (3)", "DTR (4)", "GND (5)",
        "DSR (6)", "RTS (7)", "CTS (8)", "RI (9)"
    };

    if ( pin )
        printf("%s: %s\n", function, pinname[abs(pin)+k]);
}


static void ppdebug(char *s)
{
    if (!debug) {
        if ( s == NULL )
            return;
        if ( atoi(s) != 1 )
            return;
    }

    printf("Hardware setup: %s\n"
           "Delays:\n  prog cycle = %d ms\n  power up = %d ms\n"
            "  I/O delay = %d ticks\n",
                       hw.hwname, prgdly, pwrdly, tdly);
    printf("Port %s%d (%04X) Pins:\n",
                       (hw.type==SERIAL)? "COM":"LPT", hwport,
                       (hw.type==SERIAL)? bases[hwport+4]:bases[hwport]);
    showpin("   IN", hw.inpin);
    showpin("  OUT", hw.outpin);
    showpin("  CLK", hw.clkpin);
    showpin("  VDD", hw.vddpin);
    showpin(" MCLR", hw.mclrpin);
    showpin("  PGM", hw.pgmpin);
    showpin(" READ", hw.readpin);

    if ( hwport == 0 ) {
        printf("\n\n");
        quit("Cannot debug using dummy port", FAIL);
    }

    printf("Debug mode entered ... (^C to exit) ");
    if ( getch() == 3 ) {
        printf("\n\n");
        quit(NULL,SUCCESS);
    }
    printf("\nRemove PIC ... ");
    getch();
    for(;;) {
        run_mode(STOP);
        printf("\nBegin ... ");
        if ( vdd ) {
            vddhi(1);
            printf("\nVDD on ... ");
            getch();
        }
        if ( clk ) {
            clkhi(1);
            printf("\nCLK (RB6) high ... ");
            getch();
            clklo(1);
        }
        if ( out ) {
            outhi(1);
            printf("\nOUT (RB7) high ... ");
            getch();
            outlo(0);
        }
        if ( mclr ) {
            mclrhi(1);
            printf("\nMCLR high ... ");
            getch();
            mclrlo(1);
        }
        if ( pgm ) {
            pgmhi(1);
            printf("\nPGM high ... ");
            getch();
        }
        if ( mclr && pgm ) {
            mclrhi(1);
            printf("\nPGM high, MCLR high ... ");
            getch();
            mclrlo(1);
        }
        if ( pgm )
            pgmlo(1);
        if ( in ) {
            readhi(0), outhi(1);
            printf("\nConnect a 220 ohm resistor from RB7 to VDD ... ");
            getch();
            printf("\nInput %s ... ", inbit(in)? "OK": "BAD");
            outlo(1);
            printf("\nConnect a 220 ohm resistor from RB7 to GND ... ");
            getch();
            printf("\nInput %s ... ", inbit(in)? "BAD": "OK");
            readlo(1);
        }
        run_mode(STOP);
        printf("\nStart over ... (^C to exit) ");
        if (getch() == 3) {
            printf("\n\n");
            quit(NULL,SUCCESS);
        }
    }
}

int sethws(char *s)                  /* expand descriptor if numeric */
{
    int n, nknown, id, slen;

    if ( s == NULL || (slen = strlen(s)) == 0 ) {
        hw.hws = "";
        return 1;
    }

    if ( slen >= SHWSSIZE ) {        /* assume it is not numeric */
        hw.hws = s;
        if ( hw.hwname == NULL || strlen(hw.hwname) == 0 ) {
            hw.hwname = "User defined";
            hw.hwid = -1;
        }
        return 0;
    }

    for ( n=0; n<slen; ++n )
       if ( s[n] < '0' || s[n] > '9' ) {
           hw.hws = "";
           return 1;                 /* not numeric */
       }

    n = -1;
    while ( known[++n].hws )
        ;
    nknown = n;                      /* number of known setups */

    if ( (id = atoi(s)) < 0 || id >= nknown ) {
        hw.hws = "";
        return 1;
    }

    hw.hwid =  id;
    hw.hws = known[id].hws;
    hw.hwname = known[id].hwname;

    return 0;
}


static void ppsetup(char *s)     /* get H/W setup string from environment */
{
    if ( s && (hw.hws == NULL || strlen(hw.hws) == 0) ) {
        hw.hws = s;
        hw.hwname = "User defined";
        hw.hwid = -1;            /* user defined */
    }
}


static void ppdelay(char *s)     /* get I/O delay from environment */
{
    if ( s )
        tdly = atoi(s);
}

static void ppcycle(char *s)     /* get prog cycle delay from environment */
{
    if ( prgdly == -1 && s )
        prgdly = atoi(s);
}

static void pppwrup(char *s)    /* get power up delay from environment */
{
    if ( pwrdly == -1 && s )
        pwrdly = atoi(s);
}


static void pphxfmt(char *s)     /* get dump mode from environment */
{
    if ( s )
        if ( atoi(s) == 16 )
           dumpfmt = INHX16;
}

static void ppport(char *s)  /* set port number: port 0 means no I/O */
{
    int port = 0;

    if ( hwport >= 0 )       /* hwport was set on command line */
        return;

    if ( s )
       port = atoi(s);
    else if ( (s = getenv("PPLPT")) != NULL )
       port = atoi(s);
    else if ( (s = getenv("PPCOM")) != NULL )
       port = atoi(s);
    if ( port < 0 || port > 4 )
        port = 0;
    hwport = port;
}


static int openport(void)
{
#ifdef __WIN32__
    char comport[] = "COMX";

    if ( hPort )
        CloseHandle(hPort);

    if ( hw.type == SERIAL && hwport > 0 ) {
        comport[3] = '0' + hwport;
        if ( (hPort = CreateFile(comport, GENERIC_READ | GENERIC_WRITE,
                                 0, 0, OPEN_EXISTING, 0, 0))
                                                == INVALID_HANDLE_VALUE ) {
            return 1;
        }
    }
#endif

    if ( hw.type == SERIAL ) {
        if ( hwport > 0 ) {
            io[MCR].val = inportb(io[MCR].adr);
            io[LCR].val = inportb(io[LCR].adr);
        }
    } else if ( hw.type == PARALLEL ) {
        if ( hwport > 0 ) {
            io[DATA].val = 0xFF;
            io[CONTROL].val = 0xF;
        }
    }
    return 0;
}


       /* hws2hw()
        *
        * Fills in the remainder of the HW structure from the hardware
        * descriptor string.  Returns 1 if the string is the wrong
        * length and 0 otherwise.
        *
        */

int hws2hw(HW *h)
{
    int i, pin, n;
    char *s = h->hws;

    n = (s == NULL)? 0: strlen(s);

    if ( n == 0 ) {
         h->hwid = DEFID;
         h->hws = known[DEFID].hws;
         h->hwname = known[DEFID].hwname;
         n = strlen(h->hws);
    }

    switch ( s[0] ) {
        case 's':
        case 'S': h->type = SERIAL; break;
        case 'p':
        case 'P': h->type = PARALLEL; break;
        default: h->type = PARALLEL;
    }

    if ( (h->type == PARALLEL && n != PHWSSIZE) ||
         (h->type == SERIAL && n != SHWSSIZE) )
        return 1;

    h->icsp = (s[1] == 'I')? 1: 0;  /* 'I' = in-circuit, 'P' = programmer */
    h->manual = (s[2] == 'A')? 0: 1;     /* 'A' = automatic, 'M' = manual */

    h->outpin = h->clkpin = h->inpin = 0;
    h->vddpin = h->mclrpin = h->pgmpin = h->readpin = 0;

    for ( pin=1, i=3; i<n; ++i, ++pin ) {
        switch ( s[i] ) {
            case 'I':  h->inpin = -pin; break;
            case 'i':  h->inpin = pin; break;
            case 'O':  h->outpin = -pin; break;
            case 'o':  h->outpin = pin; break;
            case 'C':  h->clkpin = -pin; break;
            case 'c':  h->clkpin = pin; break;
            case 'M':  h->mclrpin = -pin; break;
            case 'm':  h->mclrpin = pin; break;
            case 'P':  h->pgmpin = -pin; break;
            case 'p':  h->pgmpin = pin; break;
            case 'V':  h->vddpin = -pin; break;
            case 'v':  h->vddpin = pin; break;
            case 'R':  h->readpin = -pin; break;
            case 'r':  h->readpin = pin; break;
        }
    }

    return 0;
}


       /* hw2hws()
        *
        * Used by the Setup dialogue to initialise the H/W
        * descriptor string from the remainder of the HW
        * structure.
        *
        */

void hw2hws(HW *h)
{
    static char s[PHWSSIZE+1];

    memset(s, '-', PHWSSIZE);

    /* pins will be one of 0 (not defined), a pin number */
    /* or a negative pin number */

    s[abs(h->inpin)+2] = (h->inpin>0)? 'i': 'I';
    s[abs(h->outpin)+2] = (h->outpin>0)? 'o': 'O';
    s[abs(h->clkpin)+2] = (h->clkpin>0)? 'c': 'C';
    s[abs(h->vddpin)+2] = (h->vddpin>0)? 'v': 'V';
    s[abs(h->mclrpin)+2] = (h->mclrpin>0)? 'm': 'M';
    s[abs(h->pgmpin)+2] = (h->pgmpin>0)? 'p': 'P';
    s[abs(h->readpin)+2] = (h->readpin>0)? 'r': 'R';

    s[0] = (h->type == SERIAL)? 'S' : 'P';
    s[1] = (h->icsp)? 'I': 'P';
    s[2] = (h->manual)? 'M': 'A';
    s[(h->type == SERIAL)? SHWSSIZE: PHWSSIZE] = '\0';
    h->hws = s;
    h->hwname = "User defined";
    h->hwid = -1;
}


int setup(void)
{
    static int firstcall = 1;
    int i, n, iport, k, bit, base;
    int pin2bit[] = {
        0,           /* parallel port pins */
        CONTROL|BIT0,                               /* pin 1   /STROBE */
        DATA|BIT0, DATA|BIT1, DATA|BIT2, DATA|BIT3, /* pins 2-9 D0-D7 */
        DATA|BIT4, DATA|BIT5, DATA|BIT6, DATA|BIT7,
        STATUS|BIT6,                                /* pin 10  /ACK */
        STATUS|BIT7,                                /* pin 11  BUSY */
        STATUS|BIT5,                                /* pin 12  PAPEREND */
        STATUS|BIT4,                                /* pin 13  SELECT */
        CONTROL|BIT1,                               /* pin 14  /AUTOLF */
        STATUS|BIT3,                                /* pin 15  /ERROR */
        CONTROL|BIT2,                               /* pin 16  /INIT */
        CONTROL|BIT3,                               /* pin 17  /SELECTIN */

                  /* serial port pins at index 18 (NPPINS+1) to 26 */

        MSR|BIT7,                                   /* pin 1  DCD */
        0,                                          /* pin 2  RD */
        LCR|BIT6,                                   /* pin 3  TD */
        MCR|BIT0,                                   /* pin 4  DTR */
        0,                                          /* pin 5  GND */
        MSR|BIT5,                                   /* pin 6  DSR */
        MCR|BIT1,                                   /* pin 7  RTS */
        MSR|BIT4,                                   /* pin 8  CTS */
        MSR|BIT6                                    /* pin 9  RI */
    };


    valid = 0;

    if ( firstcall ) {
        giveio();                    /* request giveio.sys */
        ppsetup(getenv("PPSETUP"));
        ppport(getenv("PPPORT"));
        ppdelay(getenv("PPDELAY"));
        ppcycle(getenv("PPCYCLE"));
        pppwrup(getenv("PPPWRUP"));
        pphxfmt(getenv("PPDUMP"));
    }

    if ( sethws(hw.hws) != 0 ) {     /* hardware undefined */
        hws2hw(&hw);
        hwport = 0;
        prgdly = CYCLE;
        pwrdly = PWRDLY;
        tdly = TDLY;
        return 1;
    }

    if ( prgdly < 5 || prgdly > 50 )
        prgdly = CYCLE;

    if ( pwrdly < 5 || pwrdly > 1000 )
        pwrdly = PWRDLY;

    if ( tdly < 1 || tdly > 10000)
        tdly = TDLY;

    /* at this point hw.hws should be a hardware descriptor string */

    if ( hws2hw(&hw) != 0 )          /* initialise remainder of HW struct */
        return -1;                   /* faulty descriptor */

    can_run = hw.icsp;
    wait = hw.manual;

    k = 1;                          /* starting index for pin2bit */
    iport = hwport;                 /* index of bases */
    n = PHWSSIZE;                   /* H/W descriptor string length */
    if ( hw.type == SERIAL ) {
        k += NPPINS;                /* serial pins at offset 17 */
        n = SHWSSIZE;
        iport = hwport + 4;         /* serial bases at offset 4 */

    }

    base = bases[iport];            /* base address of I/O port */
    io[DATA].adr = base;            /* parallel port addresses */
    io[STATUS].adr = base + 1;
    io[CONTROL].adr = base + 2;
    io[LCR].adr = base + 3;         /* serial port addresses */
    io[MCR].adr = base + 4;
    io[MSR].adr = base + 6;

    if ( openport() != 0 ) {
        errmsg("Cannot open COM port");
        hwport = 0;
        return -1;
    }


    in = out = clk = mclr = pgm = vdd = read = 0;
    can_read = 0;

    for ( i=3; i<n; ++i, ++k ) {
        bit = pin2bit[k];
        switch ( hw.hws[i] ) {
            case 'I':  in = INV;
            case 'i':  in |= bit; can_read = 1; break;
            case 'O':  out = INV;
            case 'o':  out |= bit; break;
            case 'C':  clk = INV;
            case 'c':  clk |= bit; break;
            case 'M':  mclr = INV;
            case 'm':  mclr |= bit; break;
            case 'P':  pgm = INV;
            case 'p':  pgm |= bit; break;
            case 'V':  vdd = INV;
            case 'v':  vdd |= bit; break;
            case 'R':  read = INV;
            case 'r':  read |= bit; break;
            case '-':  setbit(bit); break;
            case '_':  clrbit(bit); break;
        }
    }
    if ( no_read )
        can_read = 0;

    clklo(0), outlo(0), readlo(0), pgmlo(0), vddlo(0), mclrlo(1);

    if ( firstcall )
        ppdebug(getenv("PPDEBUG"));

    firstcall = 0;
    valid = 1;
    return 0;
}

