/*
 * setup.c
 *
 * Graphical hardware setup.
 *
 * Revision history:
 *
 * 16-Nov-2001: V-0.0; created
 *
 * Copyright (C) 2001 David Tait.  All rights reserved.
 * Permission is granted to use, modify, or redistribute this software
 * so long as it is not sold or exploited for profit.
 *
 * THIS SOFTWARE IS PROVIDED AS IS AND WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED.
 *
 */

#include "fpp.h"
#ifdef __WIN32__
#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <tchar.h>
#include "resource.h"

#define MAXSTR	     256     /* size of sprintf buffer */
#define MAXNAM	     21      /* size of user assigned H/W name */
#define LPTPINS      17
#define COMPINS      9
#define NPINS	     7	     /* number of interface pins */

extern HINSTANCE hInst;      /* in main.c */
static HWND hDlg2;
static HWND hDlg1;
static TCHAR Buf[MAXSTR];    /* used by sprintf */
static HW hw_save[2];	     /* save in case user cancels H/W selection */
static int port_save[2];

enum control {
    LPT=0, COM, LPTVAL, COMVAL, OUTPIN, CLKPIN,
    INPIN, ININV, CLKINV, OUTINV, VDDPIN, VDDINV,
    MCLRPIN, MCLRINV, PGMPIN, PGMINV, READPIN,
    READINV, INVAL, OUTLO, CLKHI, CLKLO, VDDHI,
    VDDLO, MCLRHI, MCLRLO, PGMHI, PGMLO, READHI,
    READLO, APPLY, OUTHI, ICSP, MANUAL, ENABLE, NCONTROLS
};

static int iCon[] = {
    IDC_LPT, IDC_COM, IDC_LPTVAL, IDC_COMVAL, IDC_OUTPIN, IDC_CLKPIN,
    IDC_INPIN, IDC_ININV, IDC_CLKINV, IDC_OUTINV, IDC_VDDPIN, IDC_VDDINV,
    IDC_MCLRPIN, IDC_MCLRINV, IDC_PGMPIN, IDC_PGMINV, IDC_READPIN,
    IDC_READINV, IDC_INVAL, IDC_OUTLO, IDC_CLKHI, IDC_CLKLO, IDC_VDDHI,
    IDC_VDDLO, IDC_MCLRHI, IDC_MCLRLO, IDC_PGMHI, IDC_PGMLO, IDC_READHI,
    IDC_READLO, IDC_APPLY, IDC_OUTHI, IDC_ICSP, IDC_MANUAL, IDC_ENABLE
};

static HWND hCon[NCONTROLS];
static HWND hName, hPort, hPrgDly, hPwrDly, hTdly, hDefine, hDevice;

static int iPin[] = {
    OUTPIN, CLKPIN, VDDPIN, MCLRPIN, PGMPIN, READPIN, INPIN
};


static int nKnown;     /* number of known setups */
static int nDevice;    /* number of known devices */
static BOOL CanTest;


       /* CheckBox()
        *
        * Checks/unchecks a variable number of checkboxes.
        *
        */

static void CheckBox(const int n, const BOOL tick, ...)
{
    int i;
    va_list argp;
    WPARAM p = (WPARAM) BST_CHECKED;

    va_start(argp, tick);

    if ( tick == FALSE )
        p = (WPARAM) BST_UNCHECKED;

    for (i=0; i<n; ++i)
        SendMessage(hCon[va_arg(argp,int)], BM_SETCHECK, p, 0);

    va_end(argp);
}


       /* checked()
        *
        * Returns TRUE if the given checkbox is checked
        *
        */

static BOOL Checked(enum control i)
{
   if ( IsDlgButtonChecked(hDlg2, iCon[i]) == BST_CHECKED )
       return TRUE;
   return FALSE;
}

       /* Enable()
        *
        * Enables/disables a variable number of controls.
        *
        */

static void Enable(const int n, const BOOL f, ...)
{
    int i;
    va_list argp;

    va_start(argp, f);
    for (i=0; i<n; ++i)
        EnableWindow(hCon[va_arg(argp,int)], f);
    va_end(argp);
}

       /* SetMax(), SetMaxAll(), SetSel()
        *
        * Initialise selection controls.
        *
        */

static void SetMax(enum control n, int max)
{
    int i;

    SendMessage(hCon[n], LB_RESETCONTENT, 0, 0);
    SendMessage(hCon[n], LB_ADDSTRING, 0, (LPARAM) TEXT("--"));
    for ( i=1; i<=max; ++i) {
        sprintf(Buf, "%d", i);
        SendMessage(hCon[n], LB_ADDSTRING, 0, (LPARAM) Buf);
    }
    SendMessage(hCon[n], LB_SETCURSEL, 0, 0);
}

static void SetLBSel(enum control n, int index)
{
    SendMessage(hCon[n], LB_SETCURSEL, (WPARAM) index, 0);
}

static int GetLBSel(enum control n)
{
    return SendMessage(hCon[n], LB_GETTOPINDEX, 0, 0);
}

static BOOL LBAction(HWND h)
{
    return ( (h == hCon[LPTVAL]) ||
             (h == hCon[COMVAL]) ||
             (h == hCon[OUTPIN]) ||
             (h == hCon[CLKPIN]) ||
             (h == hCon[VDDPIN]) ||
             (h == hCon[MCLRPIN]) ||
             (h == hCon[PGMPIN]) ||
             (h == hCon[READPIN]) ||
             (h == hCon[INPIN]) );
}



static void SetMaxAll(void)
{
    int i;

    for ( i=0; i<NPINS; ++i )
        SetMax(iPin[i], LPTPINS);
    SetMax(LPTVAL, 3);
    SetMax(COMVAL, 4);
    SetLBSel(LPTVAL, 0);
    SetLBSel(COMVAL, 0);
    CheckBox(1, TRUE, LPT);
    Enable(1, FALSE, COMVAL);
}

       /* AdjustMax()
        *
        * Change Pin selections when new port type selected.
        *
        */

static void AdjustMax(enum control p)
{
    int i, n, max;

    if ( p == LPT ) {
        Enable(1, TRUE, LPTVAL);
        Enable(1, FALSE, COMVAL);
        max = LPTPINS;
    } else {
        Enable(1, TRUE, COMVAL);
        Enable(1, FALSE, LPTVAL);
        max = COMPINS;
    }

    for ( i=0; i<NPINS; ++i ) {
        n = SendMessage(hCon[iPin[i]], LB_GETTOPINDEX, 0, 0);
        SetMax(iPin[i], max);
        if ( n <= max )
            SetLBSel(iPin[i], n);
    }
}

static void Test(void)
{
    static int wantlo[] = {
        OUTLO, CLKLO, VDDLO, MCLRLO, PGMLO, READLO
    };
    static void (*setlo[])(int) = {
        outlo, clklo, vddlo, mclrlo, pgmlo, readlo
    };
    static void (*sethi[])(int) = {
        outhi, clkhi, vddhi, mclrhi, pgmhi, readhi
    };
    int i;

    for (i=0; i<NPINS-1; ++i)
        if ( Checked(wantlo[i]) )
            setlo[i](0);
        else
            sethi[i](0);
    assertp(1);
    us_delay(tdly);
    SetDlgItemText(hDlg2, IDC_INVAL, inval()? TEXT("High"): TEXT("Low"));
}

static void EnableTestControls(BOOL f)
{
    CanTest = FALSE;
    CheckBox(1, FALSE, ENABLE);
    if ( f ) {
        if (hw.outpin)
            Enable(2, f, OUTLO, OUTHI);
        if (hw.clkpin)
            Enable(2, f, CLKLO, CLKHI);
        if (hw.vddpin)
            Enable(2, f, VDDLO, VDDHI);
        if (hw.mclrpin)
            Enable(2, f, MCLRLO, MCLRHI);
        if (hw.pgmpin)
            Enable(2, f, PGMLO, PGMHI);
        if (hw.readpin)
            Enable(2, f, READLO, READHI);
        if (hw.inpin)
            Enable(1, f, INVAL);
        Enable(1, f, ENABLE);
    } else
        Enable(14, f, OUTLO, CLKLO, VDDLO, MCLRLO, PGMLO,
                      READLO, OUTHI, CLKHI, VDDHI, MCLRHI,
                      PGMHI, READHI, INVAL, ENABLE);
}

       /* DefFromHW()
        *
        * Define H/W, i.e. set controls, according to the current
        * HW struct.
        *
        */

static void DefFromHW(void)
{
    if ( hw.type == PARALLEL ) {
        CheckBox(1, TRUE, LPT);
        CheckBox(1, FALSE, COM);
        AdjustMax(LPT);
        SetLBSel(LPTVAL, hwport);
    } else {
        CheckBox(1, FALSE, LPT);
        CheckBox(1, TRUE, COM);
        AdjustMax(COM);
        SetLBSel(COMVAL, hwport);
    }
    if ( hw.icsp )
        CheckBox(1, TRUE, ICSP);
    if ( hw.manual)
        CheckBox(1, TRUE, MANUAL);

    SetLBSel(OUTPIN, abs(hw.outpin));
    SetLBSel(CLKPIN, abs(hw.clkpin));
    SetLBSel(VDDPIN, abs(hw.vddpin));
    SetLBSel(MCLRPIN, abs(hw.mclrpin));
    SetLBSel(PGMPIN, abs(hw.pgmpin));
    SetLBSel(READPIN, abs(hw.readpin));
    SetLBSel(INPIN, abs(hw.inpin));

    CheckBox(1, (hw.outpin<0), OUTINV);
    CheckBox(1, (hw.clkpin<0), CLKINV);
    CheckBox(1, (hw.vddpin<0), VDDINV);
    CheckBox(1, (hw.mclrpin<0), MCLRINV);
    CheckBox(1, (hw.pgmpin<0), PGMINV);
    CheckBox(1, (hw.readpin<0), READINV);
    CheckBox(1, (hw.inpin<0), ININV);
}


       /* HWFromDef()
        *
        * Use the H/W definition controls to produce a HW struct.
        *
        */

void HWFromDef(void)
{
    hw.type = Checked(COM)? SERIAL: PARALLEL;
    hw.icsp = Checked(ICSP);
    hw.manual = Checked(MANUAL);
    hw.outpin = GetLBSel(OUTPIN) * (1 - 2*Checked(OUTINV));
    hw.clkpin = GetLBSel(CLKPIN) * (1 - 2*Checked(CLKINV));
    hw.vddpin = GetLBSel(VDDPIN) * (1 - 2*Checked(VDDINV));
    hw.mclrpin = GetLBSel(MCLRPIN) * (1 - 2*Checked(MCLRINV));
    hw.pgmpin = GetLBSel(PGMPIN) * (1 - 2*Checked(PGMINV));
    hw.readpin = GetLBSel(READPIN) * (1 - 2*Checked(READINV));
    hw.inpin = GetLBSel(INPIN) * (1 - 2*Checked(ININV));
    hw2hws(&hw);
    hwport = (hw.type == SERIAL)? GetLBSel(COMVAL): GetLBSel(LPTVAL);
}


       /* DlgDefine()
        *
        * Setup hardware graphically.
        *
        */

BOOL CALLBACK DlgDefine(HWND h, UINT msg, WPARAM wParam, LPARAM lParam)
{
    int i;

    switch (msg) {

        case WM_INITDIALOG:
            hDlg2 = h;
            hw_save[1] = hw;
            port_save[1] = hwport;
            for (i=0; i<NCONTROLS; ++i)
                hCon[i] = GetDlgItem(h, iCon[i]);
            SetMaxAll();
            DefFromHW();
            CheckBox(6, TRUE, OUTLO, CLKLO, VDDLO, MCLRLO, PGMLO, READLO);
            EnableTestControls(FALSE);
            return TRUE;

        case WM_CTLCOLORLISTBOX:
             if ( LBAction((HWND) lParam) )
                 EnableTestControls(FALSE);
             return FALSE;

        case WM_COMMAND:
            switch (LOWORD(wParam)) {

                case IDOK:
                    EndDialog(h, 1);
                    return TRUE;

                case IDC_APPLY:
                    HWFromDef();
                    if ( setup() != 0 ) {
                       errmsg("Setup error");
                       EnableTestControls(FALSE);
                       return TRUE;
                    }
                    if ( hwport > 0 )
                        EnableTestControls(TRUE);
                    else
                        errmsg("No port specified");
                    return TRUE;

                case IDC_LPT:
                    AdjustMax(LPT);
                    return TRUE;

                case IDC_COM:
                    AdjustMax(COM);
                    return TRUE;

                case IDC_ICSP:
                case IDC_MANUAL:
                case IDC_OUTINV:
                case IDC_CLKINV:
                case IDC_VDDINV:
                case IDC_MCLRINV:
                case IDC_PGMINV:
                case IDC_READINV:
                case IDC_ININV:
                    EnableTestControls(FALSE);
                    return TRUE;

                case IDC_OUTLO:
                case IDC_CLKLO:
                case IDC_VDDLO:
                case IDC_MCLRLO:
                case IDC_PGMLO:
                case IDC_READLO:
                case IDC_OUTHI:
                case IDC_CLKHI:
                case IDC_VDDHI:
                case IDC_MCLRHI:
                case IDC_PGMHI:
                case IDC_READHI:
                     if ( CanTest )
                        Test();
                     return TRUE;

                case IDC_ENABLE:
                    if ( Checked(ENABLE) ) {
                        CanTest = TRUE;
                        Test();
                    } else
                        CanTest = FALSE;
                    return TRUE;

                case IDCANCEL:
                    hw = hw_save[1];
                    hwport = port_save[1];
                    EndDialog(h, 0);
                    return TRUE;

            }
            break;

        case WM_CLOSE:
                hw = hw_save[1];
                hwport = port_save[1];
                EndDialog(h, 0);
                return TRUE;

    }
    return FALSE;
}


static void SetCBSel(HWND h, int index)
{
    SendMessage(h, CB_SETCURSEL, (WPARAM) index, 0);
}


static void InitNameCB(void)
{
    int i;

    for ( i=0; known[i].hws; ++i )
        SendMessage(hName, CB_ADDSTRING, 0, (LPARAM) known[i].hwname);
    nKnown = i;
    if ( hw.hwid == -1 )
        SendMessage(hName, CB_ADDSTRING, 0, (LPARAM) hw.hwname);
    SendMessage(hName, CB_ADDSTRING, 0, (LPARAM) TEXT("Use current"));
}

static void InitPortCB(int type)
{
    int i;

    SendMessage(hPort, CB_RESETCONTENT, 0, 0);
    SendMessage(hPort, CB_ADDSTRING, 0, (LPARAM) "NULL");
    if ( type == SERIAL )
        for ( i=1; i<=4; ++i ) {
            sprintf(Buf, "COM%d  (%04XH)", i, bases[i+4]);
            SendMessage(hPort, CB_ADDSTRING, 0, (LPARAM) Buf);
        }
    else
        for ( i=1; i<=3; ++i ) {
            sprintf(Buf, "LPT%d  (%04XH)", i, bases[i]);
            SendMessage(hPort, CB_ADDSTRING, 0, (LPARAM) Buf);
        }
}

static int InitDeviceCB(void)
{
    int i, idev = -1;

    for ( i=0; device[i].part ; ++i ) {
        if ( strcmp(device[i].part, part) == 0 )
            idev = i;
        SendMessage(hDevice, CB_ADDSTRING, 0, (LPARAM) device[i].part);
    }
    nDevice = i;
    SendMessage(hDevice, CB_ADDSTRING, 0, (LPARAM) TEXT("Use current"));

    return (idev < 0)? nDevice: idev;
}


       /* DlgSetup()
        *
        * Choose Hardware, device size and timing.
        *
        */

BOOL CALLBACK DlgSetup(HWND h, UINT msg, WPARAM wParam, LPARAM lParam)
{
    static int id, type, prev_type, port, idev;
    static TCHAR HWNameBuf[MAXNAM];	 /* stores user assigned H/W name */

    switch (msg) {

        case WM_INITDIALOG:
            hDlg1 = h;
            hName = GetDlgItem(hDlg1, IDC_HWNAME);
            hPort = GetDlgItem(hDlg1, IDC_HWPORT);
            hPrgDly = GetDlgItem(hDlg1, IDC_PRGDLY);
            hPwrDly = GetDlgItem(hDlg1, IDC_PWRDLY);
            hTdly = GetDlgItem(hDlg1, IDC_TDLY);
            hDefine = GetDlgItem(hDlg1, IDC_DEFINE);
            hDevice = GetDlgItem(hDlg1, IDC_DEVICE);
            InitNameCB();
            hw_save[0] = hw;
            port_save[0] = hwport;
            if ( hw.hwid < 0 || hw.hwid > nKnown )
                SetCBSel(hName, id = nKnown);
            else
                SetCBSel(hName, id = hw.hwid);
            InitPortCB(type = hw.type);
            SetCBSel(hPort, port = hwport);
            idev = InitDeviceCB();
            SetCBSel(hDevice, idev);
            sprintf(Buf, "%d", prgdly);
            SetDlgItemText(hDlg1, IDC_PRGDLY, Buf);
            sprintf(Buf, "%d", pwrdly);
            SetDlgItemText(hDlg1, IDC_PWRDLY, Buf);
            sprintf(Buf, "%d", tdly);
            SetDlgItemText(hDlg1, IDC_TDLY, Buf);
            return TRUE;

        case WM_COMMAND:
            switch (LOWORD(wParam)) {
                case IDOK:
                    if ( id < nKnown ) {
                        hw.hwid = id;
                        hw.hws = known[id].hws;
                        hw.hwname = known[id].hwname;
                    } else {
                        GetDlgItemText(hDlg1, IDC_HWNAME, HWNameBuf, MAXNAM);
                        hw.hwname = HWNameBuf;
                    }
                    hwport = port;
                    GetDlgItemText(hDlg1, IDC_PRGDLY, Buf, 10);
                    prgdly = atoi(Buf);
                    GetDlgItemText(hDlg1, IDC_PWRDLY, Buf, 10);
                    pwrdly = atoi(Buf);
                    GetDlgItemText(hDlg1, IDC_TDLY, Buf, 10);
                    tdly = atoi(Buf);
                    idev = SendMessage(hDevice, CB_GETCURSEL, 0, 0);
                    if ( idev < nDevice ) {
                        part = device[idev].part;
                        psize = device[idev].pm;
                        dsize = device[idev].dm;
                    }
                    EndDialog(h, 1);
                    return TRUE;

                case IDC_DEFINE:
                    if ( id < nKnown ) {
                        hw.hwid = id;
                        hw.hws = known[id].hws;
                        hw.hwname = known[id].hwname;
                        hws2hw(&hw);
                    }
                    SetCBSel(hName, nKnown);
                    hwport = port;
                    DialogBox(hInst, MAKEINTRESOURCE(IDD_DEFINE),
                                                h, (DLGPROC) DlgDefine);
                    if ( hw.hwid == -1 )
                        SetCBSel(hName, id = nKnown);
                    else
                        SetCBSel(hName, id = hw.hwid);
                    InitPortCB(type = hw.type);
                    SetCBSel(hPort, port = hwport);
                    return TRUE;

               case IDC_HWNAME:
                    if ( HIWORD(wParam) == CBN_SELCHANGE ) {
                        id = SendMessage(hName, CB_GETCURSEL, 0, 0);
                        prev_type = type;
                        if ( id < nKnown )
                            type = (known[id].hws[0] == 'S')?
                                                         SERIAL: PARALLEL;
                        else
                            type = hw.type;
                        if ( type != prev_type )
                            InitPortCB(type);
                        SetCBSel(hPort, port);
                    }
                    return TRUE;

               case IDC_HWPORT:
                    if ( HIWORD(wParam) == CBN_SELCHANGE )
                        port = SendMessage(hPort, CB_GETCURSEL, 0, 0);
                    return TRUE;

                case IDCANCEL:
                    hw = hw_save[0];
                    hwport = port_save[0];
                    EndDialog(h, 0);
                    return TRUE;

            }
            break;

        case WM_CLOSE:
                hw = hw_save[0];
                hwport = port_save[0];
                EndDialog(h, 0);
                return TRUE;

    }
    return FALSE;
}

#endif /* __WIN32__ */

