* Digital Watt Meter
* Author: DPF Software Services, LLC
*
* Hardware configuration:
* =======================
*
* Antenna watt/voltage converter (Max voltage = pin voltage Vrh): PortE
*   PE0 = sensor 1 fwd
*   PE1 = sensor 1 rev
*   PE2 = sensor 2 fwd
*   PE3 = sensor 2 rev
*   PE4 = sensor 3 fwd
*   PE5 = sensor 3 rev
*   PE6 = sensor 4 fwd
*   PE7 = sensor 4 rev
*
* User interface: PortA
*  Button1..: PA0 (Mode Select, +5 released, Gnd=pressed)
*  Button2..: PA1 (data, +5 released, Gnd=pressed)
*  Relay pin: PA3 (+5 = Trip, Gnd=off)
*  Power LED: PA6 (+5 = on, Gnd=off)
*  Alarm LED: PA7 (+5 = On, Gnd=off)
*
* LCD: PortC
*
*       Uses a 2 X 16 LCD Module from Time Line Inc. 
*       Set up for 4 bit data path. 
*
*       LCD Module                              SBC-2 Port B or xicor porta
*       ----------                              ---------------------------
*       1 - Gnd                                 Gnd
*       2 - +5                                  DC+5
*       3 - Vo
*       4 - RS - 0 =instruction, 1 = data       Bit 6
*       5 - R/W - 0 = write, 1 = read           Bit 5
*       6 - Enable                              Bit 4
*       7 - D0    
*       8 - D1   
*       9 - D2     
*       10 - D3 
*       11 - D4    (4 bit path)                 Bit 0
*       12 - D5    (4 bit path)                 Bit 1
*       13 - D6    (4 bit path)                 Bit 2
*       14 - D7    (4 bit path)                 Bit 3
*
* Hardware/Software interface:
* ============================
*
*   An A/D lookup table uses 512 bytes in the program space. Values range
*   from 0-0 Watts to 255-100 Watts in 2 byte increments in an 
*   8bit integer.8bit fraction numeric format.
*
*   User settings are saved during power off periods and
*   restored at reset. This includes the current sensor selection, and user configurable options
*   for all 4 channels (bar/numberc, peak/avg, alarm, relay, type, range, and reset mode)
*
*   SWR and RELAY are operational in monitor, display, and sample modes. There is a .3 sec pause after
*   detecting an error / relay trip befor it happens.
*
*      The SWR alarm consists of a blinking dioed when
*      the SWR exceeds the user setting.
*
*      +5 volts is applied to the relay pin in order to trip the relay when the
*      the user selected SWR is exceeded.
*
*      relay and alarm errors are displayed as a user error when they are exceeded.
*
*   Button 1: Mode select.  This button cycles 
*      between the 7 operational modes.
*
*   Button 2: Data. This button modifies the select mode
*      as described.
*
*   For further mode and data descriptions see the menu specifications document
*
******************

* Currently inoperable/ problematic:
*   The A/D lookup table needs refining.
*   The automatic save/restore of the user settings.
*   All sample modes
*   The power save / xinterupt mode.
*   Error for incorrect sensor hookup (REV > FWD Watts)
*   Error for Max watts exceeded
*
***********************************************************
*
* V.1(A) Released to LDG Electronics 2/12/98.
*
*   Operational note:
*   I am still using the initial AD / WATTS lookup table that uses linear
*   interpolation between the provided readings. Based on these interpolations
*   you will get valid SWR readings with a FWD watt setting between 5 and 
*   100 Watts (slightly lower than 5 will porbably work, but it is close.) THIS
*   TABLE NEEDS CLEANED UP!
*
*   V .1(A) inoperable items:
*   =========================
*     The A/D lookup table needs refining.
*     The automatic save/restore of the user settings.
*     All sample modes
*     The power save / xinterupt mode.
*     Error for incorrect sensor hookup (REV > FWD Watts)
*     Error for Max watts exceeded
*
* V.1(B) Released to LDG Electronics 2/28
*   Changes: Source code cleaned up and better optimized.
*   user save/restore settings enabled in save debug mode.
********************************************************
* V.1CA Interm release for product testing
*
* Newly Enabled: Average routines refined, 0 sec continuous averages now
* available (default). Partial save/restore (unknown problems cause non-save
* sometimes. All sample modes enabled. power save mode (enabled in modes 1-3,7)
* Better button recognition, buttons checked 3 times each .2 second cycle.
* hardware configuration changed to SBC2-A configuration.
*
* error routine unreliable currently and disabled.
*
*******************************************************

* V.9B1 Interm release to LDG for product testing
*
* All functions should be operational
* ===================================
* Only 1 bug currently known
*    The alarm LED may flash incorrectly when buttons are held in the pressed
*    conditions.
*
* NOTE: Comments not yet complete for version .9B1
*********************************** Constant area
*
* Version .9b2
* Interim release modified for bootstrap operation (internal eeprom) startup.
* LCD timing parameters doubled (should be reviewed) to make LCD work
* Loaded the autosense minimum reading requirement to (should be reviewed)
*
***********************************
*
* Version .9b3
* Special code removed for sensor display / watt reading mode.
* Attempted temporary fix for block cursor.
*
*
**********************************
*
* Version .9b4 - Bugfix release.
*
* 1. Corrected the block cursor fix to allow the use of the block cursor 
*    during configuration setup.
* 2. Changed the auto sense mode to maintain last read channel's preferences
*    when no reading is detected. (defaults to channel one preferences
*    on bootup).
* 3. Changed the bargraph character definitions to underline the maximum
*    possible bar graph display.
* 4. Modified the bar graph display to calculate with 32 bits of precision 
*    instead of 16 bits to eliminate precision roundoff.
* 5. Changed the error and relay trip routines to "pause" .3 sec's (requiring
*    three successive bad readings) before displaying an error or tripping 
*    the relay in order to avoid inadvertant invocation due to the collapse
*    of the EMF after a transmission is complete.
* 6. Modified the setting procedure for the relay reset mode to force the
*    reset mode selection to be applied to all sensors.
* 7. Corrected an A/D conversion subsystem usage error causing intermittent 
*    reading anomolies.
**********************************
*
* Version .9b5 - Bugfix/option change release.
*
* Bugs fixed:
*   The menu will not reset while holding swr/alarm button for setting
* Option changes:
*   HF_VHF (hi/lo) -> HF (hi/lo),VHF (hi), UHF(hi)
* Other Changes:
*   Changed hf/vhf/uhf options to display the frequency ranges.
*   Changed the Hi/low options to display the power ranges.
*   Changed the 2.5 second avg to .5. 
*   Changed to a running average while gathering samples.
*
*********************************
*
* Version .9B51 - Bugfix release
*
* Bugs fixed:
*   Allowed display of the running average during initial average readings
*
*********************************
*
* Version .9B6 - Final test release
*
* Modifications made: Addition of error message when pressing the data button
* when in auto sensor mode and no channel is selected.
*
* The following pin changes have been incorporated: In addition the porta data direction registers needed
* to be set to outputs for the PA3 and PA7 pins.
*
*   Relay PA3, Pwr LED PA6, alarm LED PA7, FWD1 PE0, REV1 PE1
*   FWD2 PE2, REV2 PE3, FWD3 PE4, REV3 PE5, FWD4 PE6, REV4 PE7
*
**********************************
*
* Version .9b61 - testing bugfix release
*   corrected missing warning message
*
**********************************
*
* Version .9b62 - Change and bugfix release
*   Removed the underline from the bargraph
*   Bugfix - Removed the initial decrement on the bargraph that used to
*     compensate for round off. This is no longer necessary since roundoff 
*     error was removed and was causing one bar to be missing from the graph.
*

* m68hc11 constants
regbas     equ     $1000
eeprom     equ     $b680

porta      equ     $00
pactl      equ     $26  ddra7,paen,pamod,pedge,ddra3,i4/o5,rtr1,rtr0
portb      equ     $04
portc      equ     $03
ddrc       equ     $07  ddc7,ddc6,ddc5,ddc4,ddc3,ddc2,ddc1,ddc0
portd      equ     $08
porte      equ     $0A

tcnt       equ     $0E   main timer register
tflg2      equ     $25   tof,rtif,paovf,paif;-,-,-,-
tmsk2      equ     $24   toi,rtii,paovi,paii;-,-,pr1,pr0
option     equ     $39   adpu,csel,irqe,dly,cme,-,cr1,cf0
bprot      equ     $35   -,-,-,ptcon,bprt3,bprt2,bprt1,bprt0

* internal EEPROM constants

pprog      equ     $3B  odd,evem,elat,byte,row,erase,eelat,eepgm

* A/D converter constants

adctl      equ     $30  ccf,-,scan,mult,cd,cc,cb,ca
adr1       equ     $31
adr2       equ     $32
adr3       equ     $33
adr4       equ     $34

* LCD / ASCII interface constants

cr         equ     $0D
eot        equ     $00

* lcdport equ expporta
* lcdregbase equ expregbase
lcdport    equ     portc
lcdregbase equ     regbas

*  io interface constants

ioport     equ     porta
btnmenu    equ     $01
btndata    equ     $02
relay      equ     $08
ledon      equ     $40
ledalarm   equ     $80

* user settings eeprom storage map, channel setup constants and offsets
* This setup must mirror the initialization program routine setup
channel_data     equ  eeprom
ch_storage_size  equ  9
user_channel     equ  ch_storage_size*4+eeprom   4 channels saved.
ch_num_bar       equ  0
ch_pk_avg        equ  1
ch_alarm         equ  2
ch_relay         equ  4
ch_hf_vhf_uhf    equ  6
ch_hi_lo         equ  7
ch_man_auto      equ  8


**** Variable area
*

** LCD Display variables
linen          rmb     1

** String and decout variable
crflag         rmb     1

** wait.5 variables
wait.5cnt      rmb     2

** lookwatts variables
lookwattsret   rmb     2

** square root variables
root8.8xinter  rmb     2
root8.8oper1   rmb     2

** div 8.8 variables
num8.8den      rmb     2

** div16_16_48 bit variables
div48den       rmb     2

** swr 8.8 variables
swr8.8oper1    rmb     2

** main program variables

display_channel   rmb   1
disp_chan_hold    rmb   1
mode              rmb   1   

b1state           rmb 1
b1hold            rmb 1
b2state           rmb 1
b2hold            rmb 1

fwdhi             rmb 1
fwd               rmb 2
fwd24             rmb 3
revhi             rmb 1
rev               rmb 2
rev24             rmb 3
swr               rmb 2

alarmoncnt        rmb 1
erroroncnt        rmb 1
maincyclecnt      rmb 2

chan_read_avg     rmb 1
cur_chan_read     rmb 1

samplecnt         equ 5
avgsampcnt        rmb 1
chan_read         rmb 2*samplecnt

* This sensor channel block must remain in this order
* The program requires this data structure organization
chan_sense        equ *  (currect sensor readings)
cs_fwd1           rmb 1
cs_rev1           rmb 1
cs_fwd2           rmb 1
cs_rev2           rmb 1
cs_fwd3           rmb 1
cs_rev3           rmb 1
cs_fwd4           rmb 1
cs_rev4           rmb 1

baracchi          rmb 2
baracclo          rmb 2
barinchi          rmb 2
barinclo          rmb 2

incalarmflag      rmb 1
increlayflag      rmb 1

menuresetctr      rmb 1
menuresetmax      equ 75

man_auto_over     rmb 1

relay_reset       equ 1
relay_on          equ 2
error_fwdmax      equ 3
error_revmax      equ 4
error_rvgefw      equ 5
error_flag        rmb 1
warn_flag         rmb 1

menu_dispflag     rmb 1

sensorsleep       equ 2
sleepflag         rmb 1

relayonwait       rmb 1
*** End of Variable Area

***************************** Program code **********************************
* eprom chip starts at D000 (12K)
 org $D000


******** Subroutines ********************************************************

* LCD interface routines
*

*** set character data ram address
setCGchar:
      bsr ldispDOut
      bsr ldispDOut
      bsr ldispDOut
      bsr ldispDOut
      bsr ldispDOut
      bsr ldispDOut
      bsr ldispDOut
ldispDOut jsr dispDOut   * used to allow this portion BSR access to dispDOut
      rts

initCGram: 
      ldaa #$04
      jsr dispIOut
      clra
      jsr dispIOut
      jsr w10ms

      ldaa #$00          * bargraph blank
      bsr setCGchar
      ldaa #$10          * bargraph 1 left bar
      bsr setCGchar
      ldaa #$18          * bargraph 2 left bar
      bsr setCGchar
      ldaa #$1C          * bargraph 3 left bar
      bsr setCGchar
      ldaa #$1E          * bargraph 4 left bar
      bsr setCGchar
      ldaa #$1F          * bargraph 5 bar (full character)
      bsr setCGchar

      ldaa #$18          * do char 6 (PK)
      bsr ldispDOut
      ldaa #$14
      bsr ldispDOut
      ldaa #$18
      bsr ldispDOut
      ldaa #$10
      bsr ldispDOut
      clra
      bsr ldispDOut
      ldaa #$05
      bsr ldispDOut
      ldaa #$06
      bsr ldispDOut
      ldaa #$05
      bsr ldispDOut

      ldaa #$08          * do char7 (AV)
      bsr ldispDOut
      ldaa #$14
      bsr ldispDOut
      ldaa #$1C
      bsr ldispDOut
      ldaa #$14
      bsr ldispDOut
      clra
      bsr ldispDOut
      ldaa #$05
      bsr ldispDOut
      ldaa #$05
      bsr ldispDOut
      ldaa #$02
      bsr ldispDOut
      rts

*** Entry mode set - increment with shift
modeIS:
	psha
	ldaa    #$0
	bsr     dispIOut
	ldaa    #$6
	bsr     dispIOut
	pula
	rts

displayOnnocurs:        
	psha
	ldaa    #$0
	bsr     dispIOut
	ldaa    #$c
	bsr     dispIOut
	pula
	rts

*** function - 4 bit 2 line 5*7 character
function2LSC:
	psha
	ldaa    #$2
	bsr     dispIOut
	ldaa    #$8
	bsr     dispIOut
	pula
	rts

initIface:
	bsr     w10ms
	bsr     w10ms
	ldaa    #$3
	bsr     dispIOut
	bsr     w10ms
	ldaa    #$3
	bsr     dispIOut
	ldaa    #$3
	bsr     dispIOut
	ldaa    #$2             set 4 bit mode
	bsr     dispIOut
	bsr     function2LSC
      jsr     initCGram
	bsr     displayOnnocurs
	bsr     clear
      bsr     w10ms
	bsr     modeIS
	rts

dispDOut:
      psha
	bsr     w240mrs
	psha
	lsra
	lsra
	lsra
	lsra
	anda    #$0f
	oraa    #$40
	staa    lcdregbase+lcdport
	oraa    #$10
	staa    lcdregbase+lcdport
	anda    #$4f
	staa    lcdregbase+lcdport
	pula
	anda    #$0f
	oraa    #$40
	staa    lcdregbase+lcdport
	oraa    #$10
	staa    lcdregbase+lcdport
	anda    #$4f
	staa    lcdregbase+lcdport
      pula
	rts

dispIOut:
	bsr     w240mrs
	anda    #$0f
	staa    lcdregbase+lcdport
	oraa    #$10
	staa    lcdregbase+lcdport
	anda    #$0f
	staa    lcdregbase+lcdport
	rts
** All the LCD timing routines are doubled timing requirements to allow for variances in the LCD chips.
w10ms pshy
      ldy     #2854
      bra     wtl
w2ms  pshy
      ldy     #570
      bra     wtl
w240mrs:
	pshy
	ldy     #66
wtl   dey
	bne     wtl
	puly
	rts

*** Clear Display
clear:
	psha
	ldaa    #$0
	bsr     dispIOut
	ldaa    #$1
	bsr     dispIOut
      clr     linen
      inc     linen
	pula
	rts

*** line switch
line:
	ldaa    linen
	beq     line1
	clr     linen
	ldaa    #$40
	bsr     setDDAdd
	rts

*** first line
line1:
	ldaa    #$00
	bsr     setDDAdd
	inc     linen
	rts

*** Return home
return:
	psha
	ldaa    #$0
	bsr     dispIOut
	ldaa    #$2
	bsr     dispIOut
	pula
      clr     linen
      inc     linen
	rts

*** set Display data ram address
setDDAdd:
	oraa    #$80
	psha
	lsra
	lsra
	lsra
	lsra
	bsr     dispIOut
	pula
	bsr     dispIOut
	rts

displayOnBnocur:        
	psha
	ldaa    #$0
	bsr     dispIOut
	ldaa    #$D
	bsr     dispIOut
	pula
	rts

* WAIT ROUTINES  *************************************************************

*** Wait .5 sec
* input: Reg A = times to loop .5 seconds
* Registers unchanged by routine
* Assums a 2Mhz clock speed with a 16 bit prescaler selection

wait.5xA   pshx
	   pshb
	   psha

wait.5mlp  psha
	   ldx     #regbas
	   ldd     tcnt,x
	   addd    #62500       .5 sec tcnt offset
	   std     wait.5cnt
wait.5lp   ldd     tcnt,x       loop = 14 cycles, tcnt inc = 16 cycles
	   cpd     wait.5cnt
	   bne     wait.5lp
	   pula
	   deca
	   bne     wait.5mlp

	   pula
	   pulb
	   pulx
	   rts

* STRING OUTPUT SUBROUTINES ************************************************

*** Output string immediate with cr after text
* input: Word following jsr/bsr = address of string to display
* Output: Text displayed with CR to LCD / serial.
* Registers: Unchange by routine

osi_cr    clr    crflag         Set CR flag for outstr routine
	  inc    crflag
	  bra    osi_main

*** Output string immediate
* input: Word following jsr/bsr = address of string to display
* Output: Text displayed to LCD / serial.
* Registers: Unchange by routine

osi       clr    crflag         clear cr flag

osi_main  pshy                  save current y & x
	  pshx     
	  pshb

	  tsx                   get sp
	  ldy    5,x            get assumed return address (start of pointer to string)

	  pshy                  save pointer address
	  iny                   adjust return address to after pointer data
	  iny
	  sty    5,x            save new return address (adjust for after pointer)
	  puly                  restore pointer


	  ldy    0,y            get actual string address

	  bra    os_main_i      skip to string after register X&Y saved

*** Output string with CR 
* input: Y register contains addrsss of string to display
* Output: Text displayed with CR to LCD / serial.
* Registers: Unchange by routine

os_cr     clr    crflag
	  inc    crflag
	  bra    os_main

*** Output string with CR 
* input: Y register contains addrsss of string to display
* Output: Text displayed to LCD / serial.
* Registers: Unchanged by routine

os        clr    crflag 

os_main   pshy                  save registers
	  pshx
	  pshb
os_main_i psha

	  ldx    #regbas        setup register base address

os_lp1    ldaa   0,y            load character, old contents of a unimportant
	  beq    os_eot         (0 eot is eq when loaded into A)

	  jsr    dispDOut       LCD echo
	  iny
	  jsr    buttonchk      check buttons each character
	  bra    os_lp1         go for next character
os_eot    tst    crflag         cr or not?
	  beq    os_exit
	  jsr    line

os_exit   pula                  reset registers
	  pulb
	  pulx
	  puly
	  rts


*** Decimal Output 8.8 number with format and CR
* Input: Word following JSR/BSR contains int format (0-3) in MSB, fract format (0-3) LSB
*        D: 8.8 munber
* Output: formatted number displayed on LCD, Serial interface with a CR.
* Registers unchanged by routine.

do_cr     clr    crflag         set cr flag
	  inc    crflag
	  bra    do_main

do        clr    crflag         clear cr flag
*** Decimal Output 8.8 number with format.
* Input: byte following JSR/BSR contains int format (0-3) in bits 4-6, fract format (0-3) in bits 1-3
*        D: 8.8 munber
* Output: formatted number displayed on LCD, Serial interface.
* Registers unchanged by routine.

do_main   pshy                  save registers
	  pshx
	  pshb
	  psha


	  ldx    #'.*256       set decimal point and clear error flag
	  pshx                 get temporary string space
	  pshx                 
	  pshx                 
	  pshx                 
	  psha                 

* Stack layout:  0 1 2 3 4 5 6    7   8   9 10 11 12 13 14  15  16 
*                0 0 0 . 9 9 9 space eot  a  b  x  x  y  y ret1 ret2

	  tsy                  get string index 
	  ldaa   #3            convert 3 fractional digits
	  psha                 save for integer convert of 3 digits
do_frcvtl psha 
	  ldaa   #10           multiply fraction by 10 & save high order in ascii
	  mul
	  adda   #'0           ascii offset
	  staa   4,y           save in correct spot in string
	  iny
	  pula                 decrement loop counter, decrement, and loop if not 0
	  deca
	  bne    do_frcvtl

	  tsy                  restore string address
	  ldb    10,y          get integer
	  xgdx                 setup integer in x (a=0,b=int)
	  pulb                 convert 3 integer digits
do_incvtl pshb
	  xgdx                 restore integer part to numerator
	  ldx    #10           denominator = 10
	  idiv                                                    
	  addb   #'0           add ascii number offset to remainder and save in string
	  stb    3,y           save ascii number
	  dey                  change string offset
	  pulb                 get loop, decrement, loop if not 0.
	  decb
	  bne    do_incvtl 

	  tsy                  restore string address
	  ldx    15,y          get index to fmt masks (0=int,1=fra)

	  ldab   0,x           get integer format mask and set to 3 if to big
	  lsrb                 remove fractional format mask
	  lsrb
	  lsrb   
	  cmpb   #2
	  bls    do_fmtck1     fmt 0-2 (unsigned)
	  clrb                 use all 3 characters (3 assumed) (skip 0)
	  bra    do_chkerr     from 0-255 always ok in 3 characters
do_fmtck1 blt    do_fmtck2     fmt 0-1
	  ldab   #1            use 2 characters (skip 1)
	  ldaa   #100 
	  bra    do_fmtver
do_fmtck2 tstb                 0 or 1
	  beq    do_fmtck3     fmt=0
	  ldab   #2            use 1 character (skip 2)
	  ldaa   #10
	  bra    do_fmtver
do_fmtck3 tst    0,x           integer fmt=0, check fract fmt=0
	  bne    do_fmtck4
	  ldaa   #'*           0.0 format is invalid.
	  staa   0,y      
	  clr    1,y           set eot
	  bra    do_disp
do_fmtck4 ldab   #3            use 0 characters (skip 3)
	  ldaa   #1
do_fmtver cmpa   9,y
	  bhi    do_chkerr
do_seterr bset   8,y #255      set error flag and save
do_chkerr ldaa   8,y           check error flag
	  aby                  add offset to string pointer (set start of string for format)
	  tsta
	  beq    do_noerr
do_err    pshx                 save fmt mask address
	  tsx                  get start of string+2
	  ldaa   #'*           set error character (in D)
	  tab
	  std    3,x           can't be to big if 3 used, skip 1st char
	  std    6,x
	  staa   8,x

	  pulx                restore fmt mask address
	  ldab   0,x          get format mask
	  andb   #7           remove integer portion
	  tsx
	  bra    do_chkfra

do_noerr  ldab   0,x          get format mask
	  andb #7             remove integer portion
	  tsx                 restore start of string address
	  ldaa   #'0
	  cmpa   0,x
	  bne    do_chkdec
	  clr    0,x
	  bset   0,x #'       convert 0 to space.
	  cmpa   1,x
	  bne    do_chkdec
	  clr    1,x
	  bset   1,x #'       convert 0 to space.
do_chkdec tstb
	  bne    do_chkfra    if b not 0, find format
	  dex                 adjust for no decimal

do_chkfra cmpb   #3
	  bls    do_spceot    if fmt > 3, set as 3
	  ldab   #3

do_spceot addb   #4           inc offset to eot by 9+9+9+.
	  abx
	  ldaa   #'           space
	  staa   0,x          space end of string
	  clr    1,x          eot
do_disp   tst    crflag
	  bne    do_dispcr
	  jsr    os           display string
	  bra    do_exit
do_dispcr jsr    os_cr        display string with cr
do_exit   tsx                 get start of string
	  ldy    15,x         get rsb address
	  iny                 add 1 to ret address for fmt mask
	  sty    15,x
	
	  pula                pull string from stack
	  pulx
	  pulx
	  pulx
	  pulx

	  pula                restore registers
	  pulb 
	  pulx
	  puly
	  rts

**** Square root function
*
* Input d = number to get square root for
* Output d = square root of entry d
* Registers X,Y unchanged
*
* This routine uses newtons method with an initial quess of 8.0 and
* five iteations to return the square root of the starting value
* newtons equation: x1=(x0+rootnum/x0)/2
*
root8.8     pshx                   save registers, a&b will have result.
	    pshy
	    std   root8.8oper1     (?.? = A.B = int.fract)
	    ldd   #2048            (08.00) - initial quess for newtons method
	    std   root8.8xinter    save entry to newtons method
	    ldy   #5               5 iterations
root8.8loop ldd   root8.8oper1     oper1/xinter
	    ldx   root8.8xinter
	    jsr   num8.8div        d=numerator, x=denominator -> d=result
	    addd  root8.8xinter
	    bcc   root8.8ok
	    lsrd                   divide by 2 to get result to newtons method
	    oraa  #128             set high bit due to previous add operation overflow
	    bra   root8.8cont
root8.8ok   lsrd
root8.8cont std   root8.8xinter    save result
	    dey        
	    bne   root8.8loop      loop through newtons method
root8.8exit puly                   restore registers, a&b are result
	    pulx
	    rts


**** 16/16=48 bit divide routine
*
* Input  d = numerator (16 bit)
*        x = denominator (16 bit)
* Output d = integer result
*        x = fractional result
*        div48den = low bits 33_48 of division
 
div16_16_48 stx   div48den
	    cpd   div48den        only do high order div if necessary
	    blo   div48prefra

	    idiv                  get high order answer d/x
	    xgdx                  get result to d, remainder to x
	    psha                  save integer portion
	    pshb
	    ldd   div48den
	    xgdx                  restore remainder to d, denom to x
	    bra   div48fract

div48prefra ldx   #0              zero out integer portion
	    pshx
	    ldx   div48den

div48fract  fdiv                  only fraction by fdiv (d/x or remainder/x)

	    pshx                  save fractional result 

	    cpd div48den          calculate 33_48 bits in div32den
	    bhs div48clr48
	    ldx div48den
	    fdiv                  (rem/den)
	    stx div48den
	    bra div48ex 
div48clr48  clr div48den
	    clr div48den+1

div48ex     pulx

	    pulb                  restore integer to D, fraction in X by fdiv
	    pula
	    rts

**** 8.8 div routine
*
* Input  d = numerator (8.8 format)
*        x = denominator (8.8 format)
* Output d = result
*        x = undefined
 
num8.8div   jsr div16_16_48       divide 8.8 number by 8.8 number using div32 bit routine
	    tsta                  answer is low 8 bits of integer and upper 8 bits of fraction
	    bne num8.8ovl         if high order bits of integer non 0, overflow 8.8 number
	    pshb                  get 8.8 result into d
	    xgdx
	    tab
	    pula
	    rts
num8.8ovl   ldd #-1               set overflow
	    rts

**** SWR calculation
*
* Input d=FWD Watts (8.8 format)
*       x=REV Watts (8.8 format)
* Output d=SWR (8.8 format)
*        x=unknown
*
* routine returns answer to equation abs[(1+root(FWD/REV))/(1-root(FWD/REV))]
*
swr8.8     stx   swr8.8oper1     get rev into memory for cmp
	   cmpd  swr8.8oper1     see if rev > fwd
	   bhs   swr8.8i1
	   ldd   #$ffff          (max out since rev > fwd)
	   bra   swr8.8exit
swr8.8i1   tsta
	   bne   swr8.8i2
	   tstb
	   bne   swr8.8i2
	   inca
	   bra   swr8.8exit      fwd and rev watts = 0, swr=1 (not 1.6 as calculated)
swr8.8i2   jsr   num8.8div       FWD/REV
	   jsr   root8.8         get square root of FWD/REV
	   psha                  save rootvalue
	   pshb
	   addd  #256            add 1.0 to rootvalue (calc numerator)
	   xgdx                  d-x temp
	   pulb                  restore rootvalue
	   pula
	   pshx                  save numerator
	   cmpd  #256            since no negative number see how to get result
	   blo   swr8.8pos
	   subd  #256            subtract 1 from root (abs(1-x)=x-1 when x>1)
	   bra   swr8.8cont
swr8.8pos  std   swr8.8oper1     save in memory for subtraction
	   ldd   #256            load d with 1.0
	   subd  swr8.8oper1     done
swr8.8cont xgdx                  denominator to X
	   pula                  numerator to D
	   pulb
	   jsr   num8.8div       divide to get SWR
swr8.8exit rts                   div leaves result in D, which is SWR
	    

**** routine gets the base address of the users display channel to the y register
* input: display_channel
* output y=channel address, a,b,x unchanged

get_chanaddr  pshb
	      psha

	      ldaa display_channel  (channel-1)*size=offset from base
	      deca
	      ldab #ch_storage_size
	      mul

	      ldy    #channel_data
	      aby                         y contains base address of selected channel setup

	      pula
	      pulb
	      rts

**** watts lookup from 0-255 byte index
* 
* assumptions: each entry is 2 bytes long in the lookup tables
*              tables read from 0 to 255 in low to high order
* input  A=A/D lookup index (0=0volts=0watts / 255=2.5volts=HIWatts)
* output D=WATTS corresponding to A reading
* register x,y unchanged by routine

lookwatts     pshy
	      psha                        save original conversion reading

	      jsr    get_chanaddr         get base address of chanel into y

	      ldab   ch_hf_vhf_uhf,y      determine lookup table to use
	      bne    lookwatts_vu         Hf=0 (lo/hi),vfh=1(hi),uhf=2(hi)
	      ldab   ch_hi_lo,y
	      bne    lookwatts_h_l
	      ldy    #lookup_hf_hi
	      bra    lookwatts_get
lookwatts_h_l ldy    #lookup_hf_low
	      bra    lookwatts_get
lookwatts_vu  decb                       vhf-1=0
	      bne    lookwatts_u
	      ldy    #lookup_vhf
	      bra    lookwatts_get
lookwatts_u   ldy    #lookup_uhf         not 0(hf) or 1(vhf), must be uhf

lookwatts_get pulb                        convert original a to offset in lookup table
	      aby                         (2*0-255 reading = add twice)
	      aby
	      ldd 0,y                     return reading to D 

lookwattsex   puly
	      rts 

******** eeprom save routine
* assumed, y=eeprom address to store, a=byte to save
* return data saved in eeprom, b,x,y unchanged, a=$03
*

save_eeprom   psha
	      psha
	      ldaa  #$16            set to byte, erase, elat mode (erase byte first)
	      staa  pprog+regbas    save erase function
	      staa  0,y             identify byte
	      ldaa  #$17            byte, erase, elat, pgm mode
	      staa  pprog+regbas    save erase
	      bsr   w10ms_ee
	      dec   pprog+regbas    turn off prog
	      clr   pprog+regbas    turn off prog state and set read state

 
	      ldaa  #$02            set elat (save byte now)
	      staa  pprog+regbas
	      pula
	      staa  0,y
	      ldaa  #$03            set elat, pgm
	      staa  pprog+regbas
	      bsr   w10ms_ee
	      dec   pprog+regbas    turn off prog
	      clr   pprog+regbas    turn off prog and set read state
	      pula
	      rts

w10ms_ee      pshy                  kept this routine with eeprom in case move to ram required.
	      ldy #2856

w10ms_eelp    dey
	      bne w10ms_eelp
	      puly
	      rts
 
**** Initialize state routine
* In= x assumed to be regbase
* Out, all modes and button states initialized, alarm and relay turned off
* Register b,x,y unchanged, a=1
*
initstate   bset    ioport,x ledon         turn on power led
	    jsr     relayoff               turn off alarm and relay
	    jsr     alarmoff
	    ldaa    user_channel
	    staa    display_channel
	    bne     initstateb1
	    inc     display_channel        if user channel auto, default display channel=1
initstateb1 ldaa    #3
	    staa    alarmoncnt             display alarm for 6 revs (3on/3off)
	    staa    man_auto_over
	    staa    relayonwait
	    clra     
	    staa    mode                   mode: 0 = monitor mode
	    staa    cur_chan_read          current read channel = 0
	    staa    chan_read_avg          avg samples ok = 0, no
	    staa    incalarmflag           no alarm threshold auto increment
	    staa    increlayflag           no relay threshold auto increment
	    staa    menuresetctr           no menu reset timer
	    staa    error_flag             set no initial error
	    staa    warn_flag              turn off warning
	    inca

	    staa    menu_dispflag          start menu setting to yes
	    staa    b1state                reset buttons to not-pressed state
	    staa    b1hold                 sensor 1 selected
	    staa    b2state
	    staa    b2hold
	    rts

*** Relay / alarm setting / clearing routines
*
* x assumed to be regbase
* Output: relay alarm set/cleared as specified.
* Register a,b,y unchanged
*
relayon     bset    ioport,x relay
	    rts
relayoff    bclr    ioport,x relay
	    rts
alarmon     bset    ioport,x ledalarm
	    rts
alarmoff    bclr    ioport,x ledalarm
	    rts

**** button checking routines
*
* b1state, b2state set by routines, a = new button state, b=0
* x assumed to be regbase, 
* x,y unchanged by routines.
*

buttonchk   bsr     b1_check
	    bsr     b2_check
	    rts

b2_check    ldaa    b2hold
	    cmpa    b2state
	    bne     b2_ex                       only check if state is still the same
	    clra                                get current b2 state
	    brclr   ioport,x btndata b2_clr  
	    inca
b2_clr      sta     b2hold
b2_ex       rts


b1_check    ldaa    b1hold
	    cmpa    b1state
	    bne     b1_ex                       only check if state is still same
	    clra                                get current b1 state
	    brclr   ioport,x btnmenu b1_clr
	    inca
b1_clr      sta     b1hold
b1_ex       rts

********************************************************
* begin of main program, initialization
********************************************************

startvector equ *

	    lds     #$ff            set up stack.
	    ldx     #regbas         offset for control registers.

	    lda     option,x        set the ADPU charge pump for A/D converter
	    ora     #$80
	    staa    option,x

	    lda     tmsk2,x         set timer pre-scaler to 2Mhz/16
	    ora     #$03            @2Mhz = 125000cyc/sec = 8 micro seconds
	    sta     tmsk2,x         .5 sec = 125000/2 = 62500

	    lda     #$10            set the eeprom protection to allow writing
	    sta     bprot,x

	    ldaa    #$50            block all interrupts, allow stop
	      tap

	    lda     #$FF            Portc is for LCD output
	    sta     ddrc,x

	    lda     #$88            Porta pin 7 and pin 3 set for input
	    sta     pactl,x

	    jsr     initIface       initialize lcd interface
	    jsr     initstate       initialize main program settings

	    jsr     osi_cr
	      fdb   #pr_i1    'LDG Electronics '
	    jsr     osi_cr
	      fdb   #pr_i2    'DWM-4 Ver xxxxxx'
*           ldaa    #10
	    ldaa    #5
	    jsr     wait.5xA               10*.5

*            jsr     osi_cr
*              fdb   #pr_beta1
*            jsr     osi_cr
*              fdb   #pr_beta2
*            ldaa    #5
*            jsr     wait.5xA               5.5

	    ldd     tcnt,x
	    addd    #10                    set initial wait time to only 10 cycles
	    std     maincyclecnt

**************************************
* main program loop, timing portion
***************************************
mp_ml       ldd     maincyclecnt           check buttons till close to jump time
	    subd    tcnt,x
	    bcc     mp_mlgt
	    coma
	    comb
mp_mlgt     cmpb    #50                   if within 50 timer cycles (400 microseconds) of .1 sec, stop checking buttons
	    bls     mp_mlwt.2
	    jsr     buttonchk             3? timer cycles is typical button check time (may be longer if bounce)
	    bra     mp_ml

mp_mlwt.2   ldd     tcnt,x                cycle check is less than 1 timer cycle, catch exact cycle
	    cpd     maincyclecnt
	    bne     mp_mlwt.2

	    addd    #12500                .1 sec tcnt offset
	    std     maincyclecnt          save next time index

******************
* process buttons
******************

mp_start    ldaa    user_channel           set channel for changes to be the user channel (0 not accepted)
	    beq     mp_startncf            do not overwrite last display when auto sense mode, used later and not used for config reset
	    staa    display_channel
mp_startncf jsr     b1_check               b=?, a=current b1 state 
	    cmpa    b1state                check stabalized reading
	    beq     mp_b2                  no change in button state

	    ldab    #menuresetmax
	    stab    menuresetctr           button activity, set reset at max

	    staa    b1state                button changed state
	    beq     mp_b1prs               if state now a 0 (pressed) continue processing b1
	    clr     warn_flag              turn off warning flag (if set)
	    bra     mp_b2                  if state now a 1 (non pressed) branch
	    
mp_b1prs    bset    menu_dispflag 1        button pressed, display menu
 
	    ldaa    b2state                check for both buttons pressed
	    beq     mp_b1b2
	    ldaa    b2hold
	    bne     mp_b1change            b1 changed, b2 not now or pending change
	    staa    b2state                b2 just changed, save change
	    clr     warn_flag              disable warning, reset chosen
	    bra     mp_b1b2

mp_b1change bset    warn_flag 1            preset warning for auto change attempt
	    ldaa    user_channel           dont allow config of auto channel
	    beq     mp_b2
	    clr     warn_flag              clear warning, sensor is selected
	    inc     mode                   button state 0 (pressed), mode changed
	    ldaa    #8                     test for wrap around
	    cmpa    mode
	    bne     mp_b1ok
	    clr     mode                   wrap to mode 0 (monitor mode)
mp_b1ok     bra     mp_b2

mp_b2       jsr     b2_check               b=?,a=current b2 state
	    cmpa    b2state                check stabilized reading
	    beq     mp_b2nochg             no state change

	    bsr     mp_rsmnuctr            button activity, set reset ctr at max
	    staa    b2state                button changed state, save
	    beq     mp_b2press             if button pressed, do new b2 function
	    clr     incalarmflag           clear auto increment flags
	    clr     increlayflag
	    bra     mp_jmd

mp_b1b2     clr     man_auto_over          override relay setting
mp_resetmnu clr     mode                   reset menu timing parameters
	    clr     incalarmflag
	    clr     increlayflag
	    clr     menuresetctr 
	    jmp     mp_ml
mp_rsmnuctr ldab    #menuresetmax
	    stab    menuresetctr           
	    rts

mp_b2nochg  dec     menuresetctr           still >= 0, do not reset
	    bpl     mp_b2nccra     
	    clr     menuresetctr           set so dec does not wrap aroud
	    ldab    mode                   reset not required if already mode 0
	    bne     mp_resetmnu
mp_b2nccra  ldaa    b2state                load current state
	    bne     mp_jmd                 not pressed, skip auto increment functions

	    jsr     get_chanaddr           setup sensor loc
	    ldaa    incalarmflag           check alarm auto
	    beq     mp_b2ncrf              no, then do next autoinc
	    bsr     mp_rsmnuctr            reset menu ctr at max
	    ldd     ch_alarm,y             get current setting (alarm)
	    cmpd    #99*256                check swr of 99 max
	    beq     mp_b2ncrf                next auto function if max
	    bset    menu_dispflag 1        number change, display menu
	    addd    #32                    add .125
	    pshb                           save b portion of D
	    ldab    #ch_alarm
	    aby                            update y to be function offset of channel
	    jsr     save_eeprom            save hi portion of D
	    pula
	    iny
	    jsr     save_eeprom            save low portion of D
	    bra     mp_jmd                 done, do sensor

mp_b2ncrf   ldaa    increlayflag           check reley auto
	    beq     mp_jmd                 no, then do sensor read
	    bsr     mp_rsmnuctr            reset menu ctr at max
	    ldd     ch_relay,y             get current setting
	    cmpd    #99*256                check swr of 99 max
	    beq     mp_jmd                 at max, do sensor read
	    bset    menu_dispflag 1        number change, display menu
	    addd    #32                    add .125
	    pshb                           save b portion of d
	    ldab    #ch_relay
	    aby                            update y to be function offset of channel
	    jsr     save_eeprom            save hi portion of D
	    pula
	    iny
	    jsr     save_eeprom            save low portion of D
mp_jmd      jmp     mp_procmode            button released, no new function

mp_b2press  bset    menu_dispflag 1
	    ldaa    b1state                check for both buttons pressed
	    bne     mp_b2press1            only 1 pressed
	    clr     warn_flag              disable warning, reset chosen
	    jmp     mp_b1b2

mp_b2press1 jsr     get_chanaddr           set y as address of current display (user) channel.
mp_b2_m0    ldaa    mode                   button2 pressed, b1 released do state change
	    bne     mp_b2_m1
	    ldaa    user_channel           in mode 0, cycle channel select
	    inca
	    cmpa    #5
	    bne     mp_b2_m0b1
	    clra                           cycle to auto mode, default display=1
	    clr     display_channel
	    inc     display_channel
mp_b2_m0b1  ldy     #user_channel          save channel change to eeprom
	    jsr     save_eeprom
	    clr     chan_read_avg          reset avg readings
	    clr     cur_chan_read          and flag.
	    jmp     mp_procmode

mp_b2_m1    cmpa    #1
	    bne     mp_b2_m2
	    ldaa    ch_num_bar,y          get current setting (num/bar)
	    inca
	    cmpa    #2
	    bne     mp_b2_m1b1
	    clra
mp_b2_m1b1  ldab    #ch_num_bar
	    aby                           update y to be function offset of channel
	    jsr     save_eeprom
	    jmp     mp_procmode

mp_b2_m2    cmpa    #2
	    bne     mp_b2_m3
	    ldaa    ch_pk_avg,y          get current setting (peak/avg)
	    inca
	    cmpa    #2
	    bne     mp_b2_m2b1
	    clra
mp_b2_m2b1  ldab    #ch_pk_avg
	    aby                          update y to be function offset of channel
	    jsr     save_eeprom
	    jmp     mp_procmode


mp_b2_m3    cmpa    #3
	    bne     mp_b2_m4
	    ldd     ch_alarm,y
	    cpd     #-1                  check if alarm off
	    bne     mp_b2_m3res          if not, turn off
	    inc     incalarmflag         set autoincrement alarm flag (done by menu cycle)
	    ldd     #256+32              alarm=1.125
	    bra     mp_b2_m3sav
mp_b2_m3res ldd     #-1                  reset to off if pressing
mp_b2_m3sav pshb                         save b portion of D
	    ldab    #ch_alarm
	    aby                          update y to be function offset of channel
	    jsr     save_eeprom          save hi portion of D
	    pula
	    iny
	    jsr     save_eeprom          save low portion of D
	    jmp     mp_procmode
mp_b2_m4    cmpa    #4
	    bne     mp_b2_m5
	    ldd     ch_relay,y
	    cpd     #-1                  check if relay off
	    bne     mp_b2_m4res          if not, turn off
	    inc     increlayflag         set autoincrement alarm flag (done by menu cycle)
	    ldd     #256+32              relay=1.125
	    bra     mp_b2_m4sav
mp_b2_m4res ldd     #-1                  reset to off if pressing
mp_b2_m4sav pshb                         save b portion of D
	    ldab    #ch_relay
	    aby                          update y to be function offset of channel
	    jsr     save_eeprom          save hi portion of D
	    pula
	    iny
	    jsr     save_eeprom          save low portion of D
	    jmp     mp_procmode

mp_b2_m5    cmpa    #5
	    bne     mp_b2_m6
	    ldaa    ch_man_auto,y
	    beq     mp_b2_m5b1
	    ldaa    #-1
mp_b2_m5b1  inca
	    ldab    display_channel      save relay reset mode for all sensors
	    pshb
	    pshy

	    ldab    #ch_man_auto
	    clr     display_channel  

	    inc     display_channel      1
	    jsr     get_chanaddr  
	    aby
	    jsr     save_eeprom

	    inc     display_channel      2
	    jsr     get_chanaddr  
	    aby
	    jsr     save_eeprom

	    inc     display_channel      3
	    jsr     get_chanaddr  
	    aby
	    jsr     save_eeprom

	    inc     display_channel      4
	    jsr     get_chanaddr  
	    aby
	    jsr     save_eeprom


	    puly
	    pulb
	    stab    display_channel
	    jmp     mp_procmode

mp_b2_m6    clr     chan_read_avg        sensor change (6-7), reset avg readings
	    clr     cur_chan_read
	    cmpa    #6
	    bne     mp_b2_m7
	    ldaa    ch_hf_vhf_uhf,y          get current setting (hf/vhf/uhf)
	    inca
	    cmpa    #3
	    bne     mp_b2_m6b1
	    clra
mp_b2_m6b1  ldab    #ch_hf_vhf_uhf
	    aby                          update y to be function offset of channel
	    jsr     save_eeprom
	    jsr     get_chanaddr         reset y to base eemprom address
	    ldab    #ch_hi_lo            offset y to the hi_lo address
	    aby 
	    clra                         reset reading to HI when changing sensors
	    jsr     save_eeprom
	    jmp     mp_procmode

mp_b2_m7    ldaa    ch_hf_vhf_uhf,y
	    bne     mp_b2_m7ex           skip hi/low setting for vhf, uhf
	    ldaa    ch_hi_lo,y           get current setting (hi/low), mode 6 assumed
	    inca
	    cmpa    #2
	    bne     mp_b2_m7b1
	    clra
mp_b2_m7b1  ldab    #ch_hi_lo
	    aby                           update y to be function offset of channel
	    jsr     save_eeprom
mp_b2_m7ex  bra     mp_procmode

********************
* take readings and process
********************

mp_procmode ldx     #regbas                restore standard register base

	    ldaa    #$10                   initiate ad conversion of PE0-PE3 into ADR1-ADR4
	    staa    adctl,x
	    brclr   adctl,x #$80 *         loop until ad complete
 
	    lda     adr1,x
	    sta     cs_fwd1
	    lda     adr2,x
	    sta     cs_rev1
	    lda     adr3,x
	    sta     cs_fwd2
	    lda     adr4,x
	    sta     cs_rev2

	    ldaa    #$14                   initiate ad conversion of PE4-PE7 into ADR1-ADR4
	    staa    adctl,x
	    brclr   adctl,x #$80 *         loop until ad complete

	    lda     adr1,x
	    sta     cs_fwd3
	    lda     adr2,x
	    sta     cs_rev3
	    lda     adr3,x
	    sta     cs_fwd4
	    lda     adr4,x
	    sta     cs_rev4

	    ldab    display_channel        save last display channel
	    stab    disp_chan_hold
	    ldab    user_channel           check for auto sense
	    bne     mp_sense

	    ldaa    #sensorsleep
	    cmpa    cs_fwd1                check fwd voltage
	    bhi     mp_pm_c1r
mp_pm_c1s   ldab    #1                     simulate sensor 1
	    bra     mp_sense
mp_pm_c1r   cmpa    cs_rev1                check reverse voltage
	    bls     mp_pm_c1s
mp_pm_c2    cmpa    cs_fwd2
	    bhi     mp_pm_c2r
mp_pm_c2s   ldab    #2                     simulate sensor 2
	    bra     mp_sense
mp_pm_c2r   cmpa    cs_rev2
	    bls     mp_pm_c2s
mp_pm_c3    cmpa    cs_fwd3
	    bhi     mp_pm_c3r
mp_pm_c3s   ldab    #3                     simulate sensor 3
	    bra     mp_sense
mp_pm_c3r   cmpa    cs_rev3
	    bls     mp_pm_c3s
mp_pm_c4    ldab    #4                     simulate sensor 4 (assumed since power to at least one sensor)           

mp_sense    stab    display_channel        selected channel = displayed channel
	    decb                           offset=0-3
	    lslb                           multiply by 2 (channels) 
	    ldy     #chan_sense
	    aby                            y=chan_sense+offset for sensor pair
	    ldd     0,y
	    clr     sleepflag
	    cmpa    #sensorsleep
	    bhi     mp_sense_hi
	    cmpb    #sensorsleep
	    bhi     mp_sense_hi
	    ldaa    disp_chan_hold         return channel to last read channel before sleep
	    staa    display_channel
	    clra
	    clrb                           low reading flag and reset
	    std     0,y
	    clr     chan_read_avg          reset avg readings
	    clr     cur_chan_read          and flag.
	    inc     sleepflag
mp_sense_hi psha                           fwd
	    pshb                           rev
 
	    ldy     #chan_read
	    ldab    cur_chan_read          0-#samples

	    lslb                           multiply by 2 (channels)
	    aby                            y=chan_read+offset to channel

	    pula                           rev
	    staa    1,y                    save sensor reading(s)

	    pula                           fwd
	    staa    0,y

	    pshy                           save sensor offset (for mp_sense_cr routine)

	    ldaa     #samplecnt             get number of samples in avg
	    brset   chan_read_avg #1 mp_sense_av  if all samples taken, skip
	    ldaa    cur_chan_read          load sample count with current number
	    inca                           sample = 0-?, offset for loop, 1-(?+1)
mp_sense_av staa    avgsampcnt

	    ldaa    cur_chan_read
	    inca
	    cmpa    #samplecnt
	    bne     mp_sense_b1
	    clra
	    bset    chan_read_avg #1       if wrapped, set avg flag

mp_sense_b1 sta     cur_chan_read
	    
	    jsr     mp_sense_cr            get current sensor reading (y pulled from stack)
	    jsr     get_chanaddr  
	    brset   ch_pk_avg,y #1 mp_senseb1a branch if peak mode
	    jmp     mp_sense_go

*                                          check single reading for pk relay switch
mp_senseb1a ldd     #0
	    std     fwd24                  clear fwd and rev accumulators
	    staa    fwd24+2
	    std     rev24
	    staa    rev24+2

	    ldy     #chan_read              start at first reading
	    ldaa    avgsampcnt

mp_sense_lp psha
	    ldaa    0,y
	    jsr     lookwatts
	    addd    fwd24+1                add to cumulative fwd readings
	    std     fwd24+1
	    bcc     mp_sense_b2
	    inc     fwd24
mp_sense_b2 iny
	    ldaa    0,y

	    jsr     lookwatts
	    addd    rev24+1                add to cumulative rev readings
	    std     rev24+1
	    bcc     mp_sense_b3
	    inc     rev24
mp_sense_b3 iny
	    pula
	    deca
	    bne     mp_sense_lp  

	    clra
	    ldab    avgsampcnt             samples to divide into 16bits
	    xgdx
	    pshx                           save for other 3 div operations
	    pshx
	    pshx
	    ldd     fwd24                  2byte/1byte=1byte+remainder byte
	    idiv                           d/x = x,low + d,remain
	    stx     fwdhi                  0(hi)+1byteans(fwd)
	    tba                            1byte remainder to carry
	    ldab    fwd24+2                carry+byte in d
	    pulx                           2byte/1byte=1byte+remainder byte
	    idiv
	    xgdx
	    stb     fwd+1                  1byte ans in x(low)

	    pulx                           sample count
	    ldd     rev24                  2byte/1byte=1byte+remainder
	    idiv                           d/x = x,low + d,remain
	    stx     revhi
	    tba
	    ldab    rev24+2
	    pulx                           sample count
	    idiv
	    xgdx
	    stb     rev+1
  
	    ldx     rev
	    ldd     fwd
	    jsr     swr8.8
	    std     swr

mp_sense_go ldx     #regbas                restore standard register base after calculations
	    jsr     swrchk
	    jsr     errordisp              routine will return to mp_ml if error has been set
	    jsr     get_chanaddr           get the channel address into Y
	    ldaa    mode
	    beq     mp_m0 
	    jmp     mp_m1pt

mp_m0       jsr     displayOnnocurs
	    jsr     w2ms                   allow cursor off
	    jsr     line1
	    ldaa    ch_num_bar,y
	    beq     mp_m0_num
	    ldaa    #'F
	    jsr     dispDOut
	    ldd     fwd
	    jsr     dispbar
	    jsr     di_cfg
	    ldaa    #'R
	    jsr     dispDOut
	    ldd     rev
	    jsr     dispbar
	    bra     mp_m0_swr

mp_m0_num   jsr     osi
	      fdb   #pr_fwd
	    ldaa    sleepflag
	    beq     mp_m0_num1s           display --.- if going to sleep
	    jsr     osi
	      fdb   #pr_nosample3
	    bra     mp_m0_cfg
mp_m0_num1s ldd     fwd
	    jsr     do 
	      fcb   8*3+1
mp_m0_cfg   jsr     di_cfg
	    jsr     osi
	      fdb   #pr_rev
	    ldaa    sleepflag
	    beq     mp_m0_num2s           display --.- if going to sleep
	    jsr     osi
	      fdb   #pr_nosample3
	    bra     mp_m0_swr
mp_m0_num2s ldd     rev
	    jsr     do
	      fcb   8*3+1
	    bra     mp_m0_swr
mp_m0_swr   jsr     osi
	      fdb   #pr_s
	    ldaa    sleepflag
	    beq     mp_m0_swrs
	    jsr     osi_cr                 display --.- if going to sleep
	      fdb   #pr_nosample2
	    jmp     mp_ex
mp_m0_swrs  ldd     swr
	    jsr     do_cr
	      fcb   8*2+1
	    jmp     mp_ex

mp_m1pt     tst     menu_dispflag
	    bne     mp_m1t
	    jmp     mp_ex
mp_m1t      psha
	    clr     menu_dispflag
	    jsr     displayOnnocurs
	    jsr     w2ms                   allow cursor off
	    jsr     line1
	    pula
	    cmpa    #1
	    bne     mp_m2t
	    jsr     osi
	      fdb   #pr_display
	    jsr     di_cfg
	    jsr     osi
	      fdb   #pr_readoutmd
	    ldaa    ch_num_bar,y
	    bne     mp_m1_bar
	    jsr     osi_cr
	      fdb   #pr_num
	    bra     mp_m1_ex
mp_m1_bar   jsr     osi_cr
	      fdb   #pr_bar
mp_m1_ex    jmp     mp_ex

mp_m2t      cmpa    #2
	    bne     mp_m3t
	    jsr     osi
	      fdb   #pr_sample
	    jsr     di_cfg
	    jsr     osi
	      fdb   #pr_report
	    jsr     osi
	      fdb   #pr_mode
	    ldaa    ch_pk_avg,y
	    bne     mp_m2_bar
	    jsr     osi_cr
	      fdb   #pr_peak
	    bra     mp_m2_ex
mp_m2_bar   jsr     osi_cr
	      fdb   #pr_avg
mp_m2_ex    jmp     mp_ex

mp_m3t      cmpa    #3
	    bne     mp_m4t
	    jsr     osi
	      fdb   #pr_alarm
	    jsr     di_cfg
	    jsr     osi
	      fdb   #pr_threshold
	    ldd     ch_alarm,y
	    cmpd    #-1
	    bne     mp_m3_on
	    jsr     osi_cr
	      fdb   #pr_off
	    bra     mp_m3_ex
mp_m3_on    jsr     do_cr
	      fcb   8*2+1
mp_m3_ex    jmp     mp_ex

mp_m4t      cmpa    #4
	    bne     mp_m5t
	    jsr     osi
	      fdb   #pr_relay
	    jsr     di_cfg
	    jsr     osi
	      fdb   #pr_threshold
	    ldd     ch_relay,y
	    cmpd    #-1
	    bne     mp_m4_on
	    jsr     osi_cr
	      fdb   #pr_off
	    bra     mp_m4_ex
mp_m4_on    jsr     do_cr
	      fcb   8*2+1
mp_m4_ex    jmp     mp_ex

mp_m5t      cmpa    #5
	    bne     mp_m6t

	    jsr     osi
	      fdb   #pr_relay
	    jsr     di_cfg
	    jsr     osi
	      fdb   #pr_reset
	    jsr     osi
	      fdb   #pr_mode

	    ldaa    ch_man_auto,y
	    bne     mp_m5_auto
	    jsr     osi_cr
	      fdb   #pr_manual
	    bra     mp_m5_ex
mp_m5_auto  jsr     osi_cr
	      fdb   #pr_auto
mp_m5_ex    jmp     mp_ex

mp_m6t      cmpa    #6
	    bne     mp_m7
	    jsr     osi
	      fdb   #pr_sensor
	    jsr     di_cfg
	    jsr     osi
	      fdb   #pr_mhz
	    ldaa    ch_hf_vhf_uhf,y
	    bne     mp_m6_vu
	    jsr     osi_cr
	      fdb   #pr_hf
	    bra     mp_m6_ex
mp_m6_vu    deca
	    bne     mp_m6_uhf
	    jsr     osi_cr
	      fdb   #pr_vhf
	    bra     mp_m6_ex
mp_m6_uhf   jsr     osi_cr
	      fdb   #pr_uhf
mp_m6_ex    jmp     mp_ex

mp_m7       jsr     osi
	      fdb   #pr_sensor
	    jsr     di_cfg
	    jsr     osi
	      fdb   #pr_power
	    ldaa    ch_hi_lo,y
	    bne     mp_m7_lo
	    jsr     osi_cr
	      fdb   #pr_hi
	    bra     mp_m7_ex
mp_m7_lo    jsr     osi_cr
	      fdb   #pr_low
mp_m7_ex    jmp     mp_ex

mp_ex       ldaa    sleepflag      * if autosense mode and sleep mode, set to AUTO sense and cursor
	    beq     mp_exblock
	    ldaa    user_channel
	    bne     mp_exblock
	    ldaa    #10
	    jsr     setDDAdd        * position cursor
	    ldaa    #'A             * config = 'A-----' for autosense sleep
	    jsr     dispDOut
	    jsr     osi
	     fdb    #pr_5dash

mp_exblock  ldab    mode  * procedure to turn on block to identify mode if needed
	    cmpb    #2
	    blt     mp_exstop     * block not needed
	    lda     #9
	    adda    mode
	    cmpb    #5
	    blo     mp_exnoshft
	    deca
mp_exnoshft jsr     setDDAdd        * position cursor
	    jsr     displayOnBnocur * on block
			    
mp_exstop   ldaa    b1state     if either buttom pressed, do not
	    beq     mp_exex     stop until release is registered,
	    ldaa    b2state     otherwise exit of next stop state
	    beq     mp_exex     with pressed button will look like the release never happened. stop wanted 1-3 modes.
	    ldaa    mode
	    bne     mp_exex     don't sleep in the menu
	    ldaa    alarmoncnt
	    deca                alarm active for display 6->2
	    bpl     mp_exex     don't sleep with alarm led on
	    ldd     fwd
	    bne     mp_exex     don't sleep while a nonzero reading is displayed
	    ldd     rev
	    bne     mp_exex     don't sleep while a nonzero reading is displayed
	    ldaa    sleepflag
	    beq     mp_exex     don't sleep when instructed not to
	    stop                
mp_exex     jmp     mp_ml

dispbar:
** D=reading to display in bar format
** registers not effected
** 8 characters output representing 8 character bar+1space
**   proportional to % of max reading on sensor
	    pshy
	    pshx
	    pshb
	    psha

	    std     baracchi
	    clr     baracclo
	    clr     baracclo+1

	    jsr     get_chanaddr
	    lda     #254
	    jsr     lookwatts
	    ldx     #40

	    jsr     div16_16_48
	    std     barinchi
	    stx     barinclo

	    ldy     #barinchi           initialize constant subtractor
	    lda     #8                  8 characters
barbar      psha
	    lda     #5                  5 bars per character
	    ldx     #0                  set 0 bars as default
barchar     psha 
	    bsr     barsub              subtract constant from baracc
	    bcs     barlo2              if answer < 0, do not display bar
	    inx
barlo2      pula
	    deca
	    bne     barchar             cycle for 5 possible bars
	    xgdx                        b=character to disp (0=nobar,1=1bar,...)
	    tba                         
bardisp     jsr     dispDOut            display character
	    pula
	    deca
	    bne     barbar              cycle for 8 characters
	    ldaa    #' 
	    jsr     dispDOut            final space

	    pula
	    pulb
	    pulx 
	    puly
	    rts

**
* entry conditions: baracc >= 0, y=address of item to subtract
* exit, carry set and baracc=0 if subtraction ends with < 0, otherwise
* baracc is reduced and carry clear
**
barsub      ldd     baracclo        get lower accumulator
	    subd    2,y             subtract variable
	    std     baracclo        
	    ldd     baracchi        get higher accumulator bytes
	    bcc     barsubnobo      carry from sub instr, jump if no borrow
	    beq     barsubneg       zero from ldd instr, total ans < 0
	    subd    #1
barsubnobo  subd    0,y
	    std     baracchi
	    bcc     barsubex
barsubneg   ldd     #0              carry set if here, preserve for return
	    std     baracchi
	    std     baracclo
barsubex    rts

****** display config
* This routine displays the config setup with a CR.
* Input: None
* Output: current configuration displayed.

di_cfg      pshy
	    psha
	    pshb

	    jsr     get_chanaddr
	    ldaa    display_channel
	    adda    #'0
	    jsr     dispDOut               LCD echo
	    jsr     buttonchk      

	    ldaa #7                        AVG CGRAM character 
	    ldab    ch_pk_avg,y
	    bne     di_cfg_avg
	    ldaa #6                        PK CGRAM character
di_cfg_avg  jsr     dispDOut               LCD echo
	    jsr     buttonchk      

	    ldd     ch_alarm,y
	    cmpd    #-1
	    bne     di_cfg_aon
	    ldaa    #'*
	    bra     di_cfg_ago
di_cfg_aon  ldaa    #'A
di_cfg_ago  jsr     dispDOut              LCD echo
	    jsr     buttonchk      

	    ldd     ch_relay,y
	    cmpd    #-1
	    bne     di_cfg_ron
	    ldaa    #'*
	    bra     di_cfg_rgo
di_cfg_ron  ldaa    ch_man_auto,y
	    bne     di_cfg_rau
	    ldaa    #'R
	    bra     di_cfg_rgo
di_cfg_rau  ldaa    #'r
di_cfg_rgo  jsr     dispDOut              LCD echo
	    jsr     buttonchk      

	    ldab    ch_hf_vhf_uhf,y       0=hf, 1=vhf, 2=uhf
	    bne     di_cfg_dsvu
	    ldaa    #'H
	    bra     di_cfg_ds
di_cfg_dsvu decb
	    bne     di_cfg_dsu
	    ldaa    #'V
	    bra     di_cfg_ds
di_cfg_dsu  ldaa    #'U
di_cfg_ds   jsr     dispDOut              LCD echo
	    jsr     buttonchk      

	    ldaa    #'L
	    ldab    ch_hi_lo,y
	    bne     di_cfg_lo
	    ldaa    #'H
di_cfg_lo   jsr     dispDOut              LCD echo
	    jsr     buttonchk      
	    jsr     line

	    pulb
	    pula
	    puly
	    rts

*** One sensor reading routine
* routine takes sensor readings stored at "Y" address of sensor array
* and converts them to watts and swr values.
* Input: First stack entry = Y address of A/D readings (FWD then REV)
*        0,y = rev reading.  1,y = fwd reading
* Output: X = ?, D=swr, fwd, rev, swr set.
mp_sense_cr pulx                           ret address
	    puly
	    pshx
	    ldaa    1,y                    restore rev sensor reading(s)

	    jsr     lookwatts
	    std     rev
	    xgdx
	    ldaa    0,y

	    jsr     lookwatts
	    std     fwd                    x=rev / D = fwd
	    jsr     swr8.8
	    std     swr
	    rts

**** alarm, relay check&set; and valid sensor reading check and flag
* input: D = 8.8 SWR, relayalarm, swralarm, x=regbase, fwd, rev
* out: alarmoncnt, a=?,b=?, error_flag
*
swrchk      pshy
	    pshb
	    psha
	    ldy     mode
	    cpy     #3
	    bhi     swrchkalm
	    jsr     get_chanaddr
swrchkrly   cmpd    ch_relay,y
	    bls     swrchkless         * reset relay if in auto mode
	    ldaa    #relay_on
	    staa    error_flag
	    tst     relayonwait
	    beq     swrchkrlygo
	    dec     relayonwait
	    bra     swrchkalm
swrchkrlygo jsr     relayon
	    bra     swrchkalm
swrchkless  ldaa    #3
	    staa    relayonwait
	    ldaa    ch_man_auto,y      * only turn relay off if autorelay mode
	    bne     swrchkbauto
	    ldaa    man_auto_over      * exception set when both buttons presses (reset)
	    beq     swrchkbauto
	    brclr   ioport,x relay swrchkalm  * no error if relay not set
	    ldaa    #relay_reset       * set error when not reset
	    staa    error_flag
	    bra     swrchkalm
swrchkbauto jsr     relayoff
	    inc     man_auto_over

swrchkalm   pula                           may have skipped relay, get address & restore SWR reading
	    pulb
	    jsr     get_chanaddr          
	    cmpd    ch_alarm,y
	    bhi     swrchkb2a    
	    clra                           set alarm to start at time since no alarm
	    bra     swrchkb4

swrchkb2a   ldaa    alarmoncnt
	    deca
	    bpl     swrchkb4               alarm active for display 5->1
	    jsr     alarmon
	    bra     swrchkb5
swrchkb4    jsr     alarmoff
swrchkb5    cmpa    #-6
	    bgt     swrchkalmex
	    nega                           
swrchkalmex staa    alarmoncnt             save alarm wrapping

	    jsr get_chanaddr
	    lda #255
	    jsr lookwatts
	    cpd fwd
	    bne swrchkmaxrv
	    ldaa #error_fwdmax
	    sta error_flag
swrchkmaxrv cpd rev
	    bne swrchkrgf
	    ldaa #error_revmax
	    sta error_flag
swrchkrgf   ldd rev
	    beq swrchkex           * 0 rev = 0 fwd ok, avg sample gather
	    cpd fwd
	    blo swrchkex
	    ldaa #error_rvgefw
	    sta error_flag
swrchkex    puly
	    rts

**
** Error and warning display
**
 
errorwarn   ldaa   warn_flag
	    bne    errorwarndi
	    rts                    no warning, return
errorwarndi jsr    displayOnnocurs
	    jsr    w2ms                   allow cursor off
	    jsr    osi_cr
	      fdb  #pr_warn
	    jsr    osi_cr
	      fdb  #pr_nosensorc
	    bra    error_exml
errordisp   ldaa  error_flag
	    bne   errorchkgo
	    ldaa  #20             set error ready for next error
	    staa   erroroncnt
	    bra    errorwarn      check for warning
errorchkgo  ldaa   erroroncnt
	    bne    errorchkst
	    ldaa   #20            error wrapped, reset
errorchkst  deca
	    staa   erroroncnt
	    cmpa   #17            18,19 --> do not display (allow for instantaneous spikes at start/end of transmission)
	    blt    errorchkdil
errorchknd  clr    error_flag     error active, not displayed.
	    bra    errorwarn      check for warning
errorchkdil cmpa   #6             1,2,3,4,5 --> do not display, allow for rest of pause in cycling
	    blt    errorchknd     
	    jsr    displayOnnocurs
	    jsr    w2ms                   allow cursor off
	    jsr    osi_cr
	      fdb  #pr_error
	    ldaa   error_flag
	    cmpa   #relay_reset   
	    bne    errorchkn2t
	    jsr    osi_cr
	      fdb  #pr_relayres
	    bra    error_exml
errorchkn2t cmpa   #relay_on
	    bne    errorchkn3t
	    jsr    osi_cr
	      fdb  #pr_relayon
	    bra    error_exml
errorchkn3t cmpa   #error_fwdmax
	    bne    errorchkn4t
	    jsr    osi_cr
	      fdb  #pr_fwdmax
	    bra    error_exml
errorchkn4t cmpa   #error_revmax
	    bne    errorchkn5t
	    jsr    osi_cr
	      fdb  #pr_revmax
	    bra    error_exml
errorchkn5t cmpa   #error_rvgefw
	    bne    errorchkunk
	    jsr    osi_cr
	      fdb  #pr_rvgefw
	    bra    error_exml
errorchkunk jsr    osi_cr
	      fdb  #pr_errorunk
error_exml  pshx                   set error return to the main loop start
	    tsx
	    ldd    #mp_ml
	    std    2,x
	    pulx
	    bset   menu_dispflag 1
error_ex    clr    error_flag      clear error flag (to be set by pgm)
	    rts                    return


* lookup storage area
*   Data stored by sensor type from reading 0 to 255 in 2 byte format 8.8
*

  org $f000
lookup_hf_hi     equ *
  org $F200
lookup_hf_low    equ *
  org $f400
lookup_vhf       equ *
  org $f600
lookup_uhf       equ *

* Message Storage Area.
*    The end of text (eot[null]) character following each
*    string terminates the string, not a length or a count.  Therefore
*    the strings can be as short as desired allowing more messages
*    overall. 

 org $F900

pr_beta1     fcc  'Pre-release unit'
	     fcb  eot
pr_beta2     fcc  'Beta software   '
	     fcb  eot
pr_i1        fcc  'LDG Electronics '
	     fcb  eot
pr_i2        fcc  'DWM-4  Ver 1.00 '
	     fcb  eot
pr_fwd       fcc  'FWD:'
	     fcb  eot
pr_rev       fcc  'REV:'
	     fcb  eot
pr_s         fcc  'S:'
	     fcb  eot
pr_display   fcc  'DISPLAY   '
	     fcb  eot
pr_readoutmd fcc  'READOUT '
pr_mode      fcc  'MODE:'
	     fcb  eot
pr_bar       fcc  'BAR'
	     fcb  eot
pr_num       fcc  'NUM'
	     fcb  eot
pr_sample    fcc  'SAMPLE    '
	     fcb  eot
pr_report    fcc  'REPORT '
	     fcb  eot
pr_avg       fcc  ' AVG'
	     fcb  eot
pr_peak      fcc  'PEAK'
	     fcb  eot
pr_alarm     fcc  'ALARM     '
	     fcb  eot
pr_relay     fcc  'RELAY     '
	     fcb  eot
pr_threshold fcc  'THRESHOLD..:'
	     fcb  eot
pr_off       fcc  ' OFF'
	     fcb  eot
pr_reset     fcc  'RESET '
	     fcb  eot
pr_auto      fcc  ' AUTO'
	     fcb  eot
pr_manual    fcc  'MANUL'
	     fcb  eot
pr_sensor    fcc  'SENSOR    '
	     fcb  eot
pr_mhz       fcc  'MHZ:'
	     fcb  eot
pr_uhf       fcc  '420-460(UHF)'
	     fcb  eot
pr_vhf       fcc  '140-150(VHF)'
	     fcb  eot
pr_hf        fcc  '1.8-54 (HF+)'
	     fcb  eot
pr_power     fcc  'POWER:'
	     fcb  eot
pr_hi        fcc  '0-150W(HI)'
	     fcb  eot
pr_low       fcc  '0-15W(LOW)'
	     fcb  eot
pr_config    fcc  'CONFIG'
	     fcb  eot
pr_nosample3 fcc  '-'
pr_nosample2 fcc  '--.- '
	     fcb  eot
pr_error     fcc  '---- ERROR '
pr_5dash     fcc  '-----'
	     fcb  eot
pr_warn      fcc  '--- WARNING ----'
	     fcb  eot
pr_relayres  fcc  'RELAY RESET REQD'
	     fcb  eot
pr_relayon   fcc  'RELAY TRIPPED   '
	     fcb  eot
pr_fwdmax    fcc  'MAX FWD READING '
	     fcb  eot
pr_revmax    fcc  'MAX REV READING '
	     fcb  eot
pr_rvgefw    fcc  'REV >= FWD      '
	     fcb  eot
pr_nosensorc fcc  'NO SENSOR CHOSEN' 
	     fcb  eot
pr_errorunk  fcc  'UNKNOWN         '
	     fcb  eot

* copywrite note
 org $FE00
 fcc 'Copywrite 1998: DPF Software Services'
 
*
* use ALL RESET VECTORS just to be careful
*
 org $FFC0

 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector
 fdb #startvector

*This file contains the 2 byte lookup table.
*high byte=int, low byte (2nd line)=fract
*
* HF High


*.1 = 25
*.2 = 51
*.3 = 76
*.4 = 102
*.5 = 127
*.6 = 153
*.7 = 178
*.8 = 204
*.9 = 230


 org    61440   

 fcb    0       
 fcb    0 

 fcb    0       
 fcb    25 

 fcb    0       
 fcb    51      

 fcb    0       
 fcb    76      

 fcb    0       
 fcb    102     

 fcb    0       
 fcb    127

 fcb    0       
 fcb    153     

 fcb    0       
 fcb    178     

 fcb    0
 fcb    204     

 fcb    0       
 fcb    230     

 fcb    0       10
 fcb    230     

 fcb    1       
 fcb    0       

 fcb    1       
 fcb    51      

 fcb    1       
 fcb    102     

 fcb    1       
 fcb    127     

 fcb    1       
 fcb    178     

 fcb    1       
 fcb    230     

 fcb    2       
 fcb    0       

 fcb    2       
 fcb    51      

 fcb    2       
 fcb    102     

 fcb    2       20
 fcb    153     

 fcb    2       
 fcb    204     

 fcb    3        
 fcb    0       

 fcb    3       
 fcb    51      

 fcb    3       
 fcb    102     

 fcb    3       25
 fcb    153     

 fcb    3       
 fcb    204     

 fcb    4       
 fcb    0       

 fcb    4       
 fcb    51      

 fcb    4       
 fcb    102     

 fcb    4       30
 fcb    153     

 fcb    4       
 fcb    204     

 fcb    5       
 fcb    0       

 fcb    5       
 fcb    51      

 fcb    5       
 fcb    102     

 fcb    5       
 fcb    153     

 fcb    5       
 fcb    204     

 fcb    6       
 fcb    0       

 fcb    6       
 fcb    76      

 fcb    6       
 fcb    127     

 fcb    6       40
 fcb    204     

 fcb    7       
 fcb    0       

 fcb    7       
 fcb    51      

 fcb    7       
 fcb    127     

 fcb    7       
 fcb    204     

 fcb    8       
 fcb    0       

 fcb    8       
 fcb    76      

 fcb    8       
 fcb    153     

 fcb    9       
 fcb    0       

 fcb    9       
 fcb   76 

 fcb    9       50
 fcb   153 

 fcb    10      
 fcb   0 

 fcb    10      
 fcb   76 

 fcb    10      
 fcb   127 

 fcb    10      
 fcb    204     

 fcb    11      
 fcb   0 

 fcb    11      
 fcb   76 

 fcb    11      
 fcb   153 

 fcb    12      
 fcb   0 

 fcb    12      
 fcb   76 

 fcb    12      60
 fcb    153     

 fcb    13      
 fcb   0 

 fcb    13      
 fcb   76 

 fcb    13      
 fcb   153 

 fcb    14      
 fcb   0 

 fcb    14      
 fcb   76 

 fcb    14      
 fcb   153 

 fcb    15      
 fcb   0 

 fcb    15      
 fcb   102 

 fcb    15      
 fcb   230 

 fcb    16      70
 fcb    76      

 fcb    16      
 fcb   204 

 fcb    17      
 fcb   51 

 fcb    17      
 fcb   178 

 fcb    18      
 fcb   25 

 fcb    18      75
 fcb   153 

 fcb    19      
 fcb   0 

 fcb    19      
 fcb   127 

 fcb    20      
 fcb   0 

 fcb    20      
 fcb   127 

 fcb    20      80
 fcb    230     

 fcb    21      
 fcb   102 

 fcb    21      
 fcb   204 

 fcb    22      
 fcb   76 

 fcb    22      
 fcb   178 

 fcb    23      
 fcb   51 

 fcb    23      
 fcb   153 

 fcb    24      
 fcb   25 

 fcb    24      
 fcb   127 

 fcb    25      
 fcb   0 

 fcb    25      90
 fcb    127     

 fcb    26      
 fcb   0 

 fcb    26      
 fcb    127     

 fcb    27      
 fcb   0 

 fcb    27      
 fcb   127 

 fcb    28      
 fcb   0 

 fcb    28      
 fcb   127 

 fcb    29      
 fcb   0 

 fcb    29      
 fcb   127 

 fcb    30      
 fcb   0 

 fcb    30      100
 fcb   127 

 fcb    30      
 fcb   230 

 fcb    31      
 fcb   102 

 fcb    31      
 fcb   230 

 fcb    32      
 fcb   76 

 fcb    32      
 fcb   204 

 fcb    33      
 fcb   51 

 fcb    33      
 fcb   178 

 fcb    34      
 fcb   25 

 fcb    34      
 fcb   127 

 fcb    35      110
 fcb    0       

 fcb    35      
 fcb   127 

 fcb    36      
 fcb   0 

 fcb    36      
 fcb   127 

 fcb    37      
 fcb   0 

 fcb    37      
 fcb   127 

 fcb    38      
 fcb   0 

 fcb    38      
 fcb   127 

 fcb    39      
 fcb   0 

 fcb    39      
 fcb   127 

 fcb    40      120
 fcb    0       

 fcb    40      
 fcb   127 

 fcb    41      
 fcb   0 

 fcb    41      
 fcb   153 

 fcb    42      
 fcb   51 

 fcb    42      125
 fcb   178
 
 fcb    43      
 fcb   76 

 fcb    43      
 fcb   204 

 fcb    44      
 fcb   102 

 fcb    45      
 fcb    0       

 fcb    45      130
 fcb    153     

 fcb    46      
 fcb   51 

 fcb    46      
 fcb   204 

 fcb    47      
 fcb   127 

 fcb    48      
 fcb   25 

 fcb    48      
 fcb   204 

 fcb    49      
 fcb   102 

 fcb    50      
 fcb   0 

 fcb    50      
 fcb   178 

 fcb    51      
 fcb   102

 fcb    52      140
 fcb    25      

 fcb    52 
 fcb    204

 fcb    53 
 fcb    127

 fcb    54 
 fcb    51

 fcb    55 
 fcb    0

 fcb    55 
 fcb    178

 fcb    56 
 fcb    102

 fcb    57 
 fcb    25

 fcb    57 
 fcb    204

 fcb    58 
 fcb    127

 fcb    59     150
 fcb    51

 fcb    60
 fcb   0 

 fcb    60 
 fcb   178

 fcb    61 
 fcb    102

 fcb    62 
 fcb   25

 fcb    62 
 fcb   204

 fcb    63 
 fcb   127

 fcb    64 
 fcb   51

 fcb    65 
 fcb   0 

 fcb    65 
 fcb   178

 fcb    66     160
 fcb    76

 fcb    67 
 fcb   0 

 fcb    67 
 fcb   178

 fcb    68 
 fcb   76

 fcb    69
 fcb   25

 fcb    70 
 fcb    0       

 fcb    70 
 fcb   204 

 fcb    71 
 fcb   153 

 fcb    72 
 fcb   102

 fcb    73 
 fcb   76 

 fcb    74     170
 fcb    51

 fcb    75 
 fcb   0 
 
 fcb    75 
 fcb   178 
 
 fcb    76 
 fcb   102 
 
 fcb    77 
 fcb   25 
 
 fcb    77     175
 fcb   230 
 
 fcb    78 
 fcb   153
 
 fcb    79 
 fcb    76 

 fcb    80 
 fcb   0 
 
 fcb    80 
 fcb   204 
 
 fcb    81     180
 fcb    178

 fcb    82 
 fcb   127 
 
 fcb    83 
 fcb   76
 
 fcb    84 
 fcb   25
 
 fcb    85 
 fcb   0 
 
 fcb    85 
 fcb   204
 
 fcb    86 
 fcb   127
 
 fcb    87 
 fcb   127
 
 fcb    88 
 fcb   76
 
 fcb    89
 fcb   25

 fcb    90     190
 fcb    0       

 fcb    90 
 fcb   178
 
 fcb    91 
 fcb   127
 
 fcb    92 
 fcb   76
 
 fcb    93 
 fcb   25
 
 fcb    93 
 fcb   230
 
 fcb    95 
 fcb   0 
 
 fcb    95 
 fcb   204
 
 fcb    96 
 fcb   153
 
 fcb    97 
 fcb   102
 
 fcb    98     200
 fcb   51
 
 fcb    99 
 fcb   0 

 fcb    100 
 fcb   0 
 
 fcb    100 
 fcb   230 
 
 fcb    101 
 fcb   204 
 
 fcb    102 
 fcb   178
 
 fcb    103 
 fcb   153
 
 fcb    104 
 fcb   127
 
 fcb    105 
 fcb   102
 
 fcb    106 
 fcb   76 
 
 fcb    107     210
 fcb    51

 fcb    108 
 fcb   25
 
 fcb    109 
 fcb   0 
 
 fcb    110 
 fcb   0 

 fcb   111  
 fcb   25 
 
 fcb    112
 fcb   51
 
 fcb    113 
 fcb   102
 
 fcb    114 
 fcb   127
 
 fcb    115 
 fcb   153
 
 fcb    116 
 fcb   178
 
 fcb    117     220
 fcb    204

 fcb    118 
 fcb   230
 
 fcb    120 
 fcb   0 
 
 fcb    121 
 fcb   51
 
 fcb    122 
 fcb   127
 
 fcb    123     225
 fcb   178
 
 fcb    125 
 fcb   0
 
 fcb    126 
 fcb   51
 
 fcb    127 
 fcb   127
 
 fcb    128 
 fcb   178
 
 fcb    130     230
 fcb    0       

 fcb    131 
 fcb   0 
 
 fcb    132 
 fcb   0 
 
 fcb    133 
 fcb   0 
 
 fcb    134 
 fcb   0 
 
 fcb    134 
 fcb   230 
 
 fcb    135 
 fcb   204
 
 fcb    136 
 fcb   178
 
 fcb    137 
 fcb    230

 fcb    139
 fcb   0 
 
 fcb    140     240
 fcb    0       

 fcb    141 
 fcb   0 
 
 fcb    142 
 fcb   0 
 
 fcb    143 
 fcb   0 
 
 fcb    144 
 fcb   0 
 
 fcb    145 
 fcb   0 
 
 fcb    146 
 fcb   0 
 
 fcb    147 
 fcb   0 
 
 fcb    148 
 fcb   0 
 
 fcb    149 
 fcb   0 
 
 fcb    150     250
 fcb    0       

 fcb    151 
 fcb   0 
 
 fcb    152 
 fcb   0 
 
 fcb    153 
 fcb   0 
 
 fcb    154 
 fcb   0 
 
 fcb    155     255
 fcb    0

*************************************
*HF low
 
 org   61952 

 fcb    0
 fcb    0 

 fcb    0       
 fcb    25 

 fcb    0       
 fcb    25 

 fcb    0       
 fcb    25 

 fcb    0       
 fcb    25  

 fcb    0       
 fcb    51 

 fcb    0       
 fcb    51  

 fcb    0       
 fcb    51  

 fcb    0
 fcb    51  

 fcb    0       
 fcb    51  

 fcb    0       10
 fcb    76  

 fcb    0 
 fcb    76

 fcb    0 
 fcb    76 

 fcb    0 
 fcb    76 

 fcb    0 
 fcb    102 

 fcb    0
 fcb    102 

 fcb    0 
 fcb    102 

 fcb    0 
 fcb    102

 fcb    0 
 fcb    127

 fcb    0 
 fcb    127 

 fcb    0       20
 fcb    127 

 fcb    0 
 fcb    127 

 fcb    0        
 fcb    153

 fcb    0 
 fcb    153

 fcb    0 
 fcb    153 

 fcb    0       25
 fcb    153     

 fcb    0 
 fcb    178 

 fcb    0 
 fcb    178

 fcb    0 
 fcb    178

 fcb    0 
 fcb    204 

 fcb    0       30
 fcb    204 

 fcb    0
 fcb    204     

 fcb    0 
 fcb    230

 fcb    0 
 fcb    230

 fcb    0 
 fcb    230 

 fcb    1 
 fcb    0 

 fcb    1 
 fcb    0 

 fcb    1 
 fcb    0       

 fcb    1 
 fcb    0  

 fcb    1 
 fcb    0   

 fcb    1       40
 fcb    25

 fcb    1 
 fcb    25

 fcb    1
 fcb    25

 fcb    1
 fcb    25

 fcb    1
 fcb    51

 fcb    1
 fcb    51

 fcb    1
 fcb    51

 fcb    1
 fcb    51

 fcb    1
 fcb    76

 fcb    1
 fcb    76

 fcb    1      50
 fcb    76

 fcb    1
 fcb    102

 fcb    1 
 fcb    102

 fcb    1
 fcb   102

 fcb    1
 fcb    127

 fcb    1
 fcb    127

 fcb    1 
 fcb   127

 fcb    1 
 fcb   127

 fcb    1 
 fcb   153

 fcb    1 
 fcb   153

 fcb    1      60
 fcb    153     

 fcb    1 
 fcb    153

 fcb    1
 fcb   178

 fcb    1 
 fcb   178

 fcb    1 
 fcb   178

 fcb    1 
 fcb   204

 fcb    1 
 fcb   204

 fcb    1 
 fcb   204

 fcb    1 
 fcb   230

 fcb    1 
 fcb   230 

 fcb    1     70
 fcb    230

 fcb    2
 fcb   0

 fcb    2
 fcb   0

 fcb    2
 fcb   0

 fcb    2
 fcb   25

 fcb    2     75
 fcb   25

 fcb    2
 fcb   25

 fcb    2
 fcb   51

 fcb    2 
 fcb   51

 fcb    2 
 fcb   76

 fcb    2      80
 fcb    76

 fcb    2
 fcb    76

 fcb    2
 fcb   102

 fcb    2 
 fcb   102

 fcb    2 
 fcb   127

 fcb    2 
 fcb   127

 fcb    2 
 fcb   127

 fcb    2 
 fcb   153

 fcb    2 
 fcb   153

 fcb    2 
 fcb   178

 fcb    2      90
 fcb    178

 fcb    2
 fcb   178

 fcb    2 
 fcb    204

 fcb    2
 fcb   204

 fcb    2 
 fcb   230

 fcb    2 
 fcb   230

 fcb    2
 fcb   230

 fcb    3
 fcb   0 

 fcb    3
 fcb   0

 fcb    3 
 fcb   25

 fcb    3      100
 fcb   25

 fcb    3 
 fcb   51

 fcb    3 
 fcb   51

 fcb    3 
 fcb   76

 fcb    3 
 fcb   76

 fcb    3 
 fcb   102

 fcb    3 
 fcb   102

 fcb    3 
 fcb   127

 fcb    3 
 fcb   127

 fcb    3 
 fcb   153

 fcb    3      110
 fcb    153

 fcb    3
 fcb   178

 fcb    3 
 fcb   178

 fcb    3 
 fcb   204

 fcb    3 
 fcb   204

 fcb    3 
 fcb   230

 fcb    3 
 fcb   230

 fcb    4
 fcb   0

 fcb    4
 fcb   0 

 fcb    4
 fcb   25

 fcb    4      120
 fcb    51

 fcb    4 
 fcb   51

 fcb    4 
 fcb   76

 fcb    4 
 fcb   76

 fcb    4 
 fcb   102

 fcb    4      125
 fcb   127
 
 fcb    4 
 fcb   127

 fcb    4 
 fcb   153

 fcb    4 
 fcb   153

 fcb    4 
 fcb    178

 fcb    4      130
 fcb    204

 fcb    4 
 fcb   204

 fcb    4 
 fcb   230

 fcb    5
 fcb   0

 fcb    5
 fcb   0

 fcb    5
 fcb   25

 fcb    5
 fcb   25

 fcb    5 
 fcb   51

 fcb    5 
 fcb   76

 fcb    5
 fcb   76

 fcb    5      140
 fcb    102

 fcb    5
 fcb    127

 fcb    5
 fcb    127

 fcb    5
 fcb    153

 fcb    5
 fcb    178

 fcb    5 
 fcb    178

 fcb    5
 fcb    204

 fcb    5
 fcb    230

 fcb    5
 fcb    230

 fcb    6
 fcb    0

 fcb    6    150
 fcb    25

 fcb    6
 fcb   51

 fcb    6
 fcb   51

 fcb    6 
 fcb    76

 fcb    6
 fcb   102

 fcb    6 
 fcb   127

 fcb    6 
 fcb   127

 fcb    6 
 fcb   153

 fcb    6 
 fcb   178

 fcb    6 
 fcb   204

 fcb    6     160
 fcb    230

 fcb    6
 fcb   230

 fcb    7
 fcb   0

 fcb    7
 fcb   25

 fcb    7
 fcb   51

 fcb    7 
 fcb    51

 fcb    7 
 fcb   76

 fcb    7 
 fcb   102

 fcb    7 
 fcb   127

 fcb    7 
 fcb   127

 fcb    7     170
 fcb    153

 fcb    7
 fcb   178
 
 fcb    7 
 fcb   204
 
 fcb    7 
 fcb   204
 
 fcb    7 
 fcb   230
 
 fcb    8    175
 fcb   0
 
 fcb    8
 fcb   25
 
 fcb    8
 fcb    25

 fcb    8 
 fcb   51
 
 fcb    8 
 fcb   76
 
 fcb    8     180
 fcb    102

 fcb    8
 fcb   102
 
 fcb    8
 fcb   127
 
 fcb    8 
 fcb   153
 
 fcb    8 
 fcb   178
 
 fcb    8 
 fcb   178
 
 fcb    8 
 fcb   204
 
 fcb    8
 fcb   230
 
 fcb    9
 fcb   0
 
 fcb    9
 fcb   25

 fcb    9     190
 fcb    51

 fcb    9 
 fcb   76
 
 fcb    9 
 fcb   102
 
 fcb    9 
 fcb   102
 
 fcb    9 
 fcb   127
 
 fcb    9 
 fcb   153
 
 fcb    9 
 fcb   178
 
 fcb    9
 fcb   178
 
 fcb    9 
 fcb   204
 
 fcb    9 
 fcb   230
 
 fcb    10   200
 fcb   0
 
 fcb    10
 fcb   25

 fcb    10 
 fcb   51
 
 fcb    10 
 fcb   76
 
 fcb    10 
 fcb   102
 
 fcb    10 
 fcb   127
 
 fcb    10 
 fcb   127
 
 fcb    10 
 fcb   153
 
 fcb    10 
 fcb   178
 
 fcb    10 
 fcb   204
 
 fcb    10     210
 fcb    230

 fcb    11
 fcb   0
 
 fcb    11
 fcb   25
 
 fcb    11 
 fcb   51

 fcb   11  
 fcb   76
 
 fcb    11
 fcb   102
 
 fcb    11 
 fcb   127
 
 fcb    11 
 fcb   153
 
 fcb    11 
 fcb   178
 
 fcb    11 
 fcb   204
 
 fcb    11     220
 fcb    230

 fcb    12
 fcb   0
 
 fcb    12 
 fcb   25
 
 fcb    12 
 fcb   51
 
 fcb    12 
 fcb   76
 
 fcb    12     225
 fcb   102
 
 fcb    12 
 fcb   127
 
 fcb    12 
 fcb   153
 
 fcb    12 
 fcb   178
 
 fcb    12 
 fcb   204
 
 fcb    12    230
 fcb    230

 fcb    13 
 fcb   0 
 
 fcb    13 
 fcb   25
 
 fcb    13 
 fcb   51
 
 fcb    13 
 fcb   76
 
 fcb    13 
 fcb   102
 
 fcb    13 
 fcb   153
 
 fcb    13 
 fcb   178
 
 fcb    13 
 fcb    204

 fcb    13
 fcb   230
 
 fcb    14     240
 fcb    0       

 fcb    14 
 fcb   25
 
 fcb    14 
 fcb   51
 
 fcb    14 
 fcb   76
 
 fcb    14 
 fcb   102
 
 fcb    14 
 fcb   127
 
 fcb    14 
 fcb   153
 
 fcb    14 
 fcb   178
 
 fcb    14 
 fcb   204
 
 fcb    14 
 fcb   230
 
 fcb    15     250
 fcb    0       

 fcb    15 
 fcb   25
 
 fcb    15 
 fcb   51
 
 fcb    15 
 fcb   76
 
 fcb    15 
 fcb   102
 
 fcb    15     255
 fcb    153

***************************************
*VHF

 org    62464   

 fcb    0
 fcb    0 

 fcb    0       
 fcb    25 

 fcb    0       
 fcb    25 

 fcb    0       
 fcb    25 

 fcb    0       
 fcb    51 

 fcb    0       
 fcb    51

 fcb    0       
 fcb    51 

 fcb    0       
 fcb    76 

 fcb    0
 fcb    76 

 fcb    0       
 fcb    102 

 fcb    0       10
 fcb    102 

 fcb    0 
 fcb    127 

 fcb    0
 fcb    127 

 fcb    0
 fcb    153 

 fcb    0
 fcb    178 

 fcb    0
 fcb    204 

 fcb    0
 fcb    230 

 fcb    1
 fcb    0       

 fcb    1 
 fcb    25 

 fcb    1
 fcb    76 

 fcb    1       20
 fcb    102 

 fcb    1 
 fcb    127 

 fcb    1        
 fcb    153 

 fcb    1
 fcb    178

 fcb    1
 fcb    204 

 fcb    1       25
 fcb    230 

 fcb    2 
 fcb    0 

 fcb    2 
 fcb    51 

 fcb    2
 fcb    76 

 fcb    2
 fcb    127 

 fcb    2     30
 fcb    178 

 fcb    2 
 fcb    204     

 fcb    3 
 fcb    0       

 fcb    3 
 fcb    51      

 fcb    3 
 fcb    102     

 fcb    3 
 fcb    153     

 fcb    3 
 fcb    204     

 fcb    4 
 fcb    0       

 fcb    4 
 fcb    51 

 fcb    4 
 fcb    127     

 fcb    4       40
 fcb    178 

 fcb    5 
 fcb    0       

 fcb    5 
 fcb    51      

 fcb    5 
 fcb    127     

 fcb    5 
 fcb    178 

 fcb    6 
 fcb    0       

 fcb    6 
 fcb    51 

 fcb    6
 fcb    127 

 fcb    6
 fcb    178

 fcb    7
 fcb   0 

 fcb    7       50
 fcb   76 

 fcb    7 
 fcb   153

 fcb    8 
 fcb   0 

 fcb    8 
 fcb   76 

 fcb    8 
 fcb    153 

 fcb    9
 fcb   0 

 fcb    9 
 fcb   51 

 fcb    9 
 fcb   127 

 fcb    9 
 fcb   178 

 fcb    10 
 fcb   0

 fcb    10      60
 fcb    76 

 fcb    10 
 fcb   153

 fcb    11 
 fcb   0

 fcb    11 
 fcb   127 

 fcb    12 
 fcb   0 

 fcb    12 
 fcb   127

 fcb    13 
 fcb   0 

 fcb    15 
 fcb   127

 fcb    14 
 fcb   0 

 fcb    14 
 fcb   76 

 fcb    14      70
 fcb    153 

 fcb    15 
 fcb   0 

 fcb    15
 fcb   127

 fcb    16 
 fcb   0 

 fcb    16 
 fcb   127

 fcb    17      75
 fcb   0 

 fcb    17 
 fcb   127 

 fcb    18 
 fcb   0 

 fcb    18 
 fcb   127

 fcb    19 
 fcb   0 

 fcb    19      80
 fcb    127 

 fcb    20
 fcb   0 

 fcb    20 
 fcb   127 

 fcb    21 
 fcb   25

 fcb    21 
 fcb   153 

 fcb    22 
 fcb   51 

 fcb    22 
 fcb   178 

 fcb    23 
 fcb   76

 fcb    23 
 fcb   204 

 fcb    24 
 fcb   102

 fcb    24      90
 fcb    230 

 fcb    25 
 fcb   127

 fcb    26
 fcb    0 

 fcb    26 
 fcb   153

 fcb    27 
 fcb   25 

 fcb    27 
 fcb   178

 fcb    28 
 fcb   51 

 fcb    28 
 fcb   204

 fcb    29      
 fcb   102 

 fcb    30      
 fcb   0 

 fcb    30      100
 fcb   153 

 fcb    31 
 fcb   25 

 fcb    31 
 fcb   178 

 fcb    32 
 fcb   76 

 fcb    32      
 fcb   230

 fcb    33 
 fcb   127 

 fcb    34 
 fcb   0

 fcb    34 
 fcb   153 

 fcb    35 
 fcb   25

 fcb    35 
 fcb   178 

 fcb    36      110
 fcb    76

 fcb    36 
 fcb   230 

 fcb    37 
 fcb   127

 fcb    38 
 fcb   25 

 fcb    38 
 fcb   178

 fcb    39 
 fcb   102 

 fcb    40 
 fcb   0 

 fcb    40 
 fcb   178 

 fcb    41 
 fcb   102

 fcb    42 
 fcb   25 

 fcb    42      120
 fcb    204

 fcb    43
 fcb   127 

 fcb    44 
 fcb   51

 fcb    45 
 fcb   0 

 fcb    45 
 fcb   178

 fcb    46      125
 fcb   51
 
 fcb    46 
 fcb   230

 fcb    47 
 fcb   178 

 fcb    48 
 fcb   127 

 fcb    49 
 fcb    76

 fcb    50      130
 fcb    0 

 fcb    50 
 fcb   204

 fcb    51 
 fcb   153 

 fcb    52 
 fcb   76 

 fcb    53 
 fcb   25

 fcb    53 
 fcb   230 

 fcb    54 
 fcb   178 

 fcb    55 
 fcb   127

 fcb    56 
 fcb   76 

 fcb    57 
 fcb   25

 fcb    57      140
 fcb    204 

 fcb    58
 fcb    127

 fcb    59
 fcb    76

 fcb    60 
 fcb    0

 fcb    60 
 fcb    230

 fcb    61
 fcb    204

 fcb    62
 fcb    178

 fcb    63
 fcb    153

 fcb    64
 fcb    127

 fcb    65
 fcb    104

 fcb    66     150
 fcb    76

 fcb    67
 fcb   51

 fcb    68 
 fcb   25

 fcb    69 
 fcb    0

 fcb    70 
 fcb   0

 fcb    70 
 fcb   204

 fcb    71 
 fcb   153

 fcb    72 
 fcb   102

 fcb    73 
 fcb   51 

 fcb    74 
 fcb   25

 fcb    75     160
 fcb    0

 fcb    75 
 fcb   204

 fcb    76 
 fcb   153

 fcb    77 
 fcb   102

 fcb    78
 fcb   51

 fcb    79 
 fcb    25

 fcb    80 
 fcb   0 

 fcb    80 
 fcb   230 

 fcb    81 
 fcb   204

 fcb    82 
 fcb   178

 fcb    83     170
 fcb    153

 fcb    84
 fcb   127
 
 fcb    85 
 fcb   102 
 
 fcb    86 
 fcb   76 
 
 fcb    87 
 fcb   51
 
 fcb    88     175
 fcb   25 
 
 fcb    89 
 fcb   0
 
 fcb    90 
 fcb    0 

 fcb    90 
 fcb   230
 
 fcb    91 
 fcb   204 
 
 fcb    92     180
 fcb    178

 fcb    93
 fcb   153 
 
 fcb    94 
 fcb   127
 
 fcb    95 
 fcb   102
 
 fcb    96 
 fcb   76
 
 fcb    97 
 fcb   51
 
 fcb    98 
 fcb   25
 
 fcb    99 
 fcb   0
 
 fcb    100 
 fcb   0
 
 fcb    101
 fcb   0

 fcb    102     190
 fcb    0       

 fcb    103 
 fcb   0
 
 fcb    104 
 fcb   0
 
 fcb    105 
 fcb   0
 
 fcb    106 
 fcb   0
 
 fcb    107 
 fcb   0
 
 fcb    108 
 fcb   0 
 
 fcb    109 
 fcb   0
 
 fcb    110 
 fcb   0
 
 fcb    111 
 fcb   0
 
 fcb    112     200
 fcb   0
 
 fcb    113 
 fcb   0 

 fcb    114 
 fcb   0 
 
 fcb    115 
 fcb   0
 
 fcb    116 
 fcb   0
 
 fcb    117 
 fcb   0
 
 fcb    118 
 fcb   0
 
 fcb    119 
 fcb   0
 
 fcb    120 
 fcb   0
 
 fcb    121 
 fcb   25
 
 fcb    122     210
 fcb    76

 fcb    123 
 fcb   102
 
 fcb    124 
 fcb   127
 
 fcb    125 
 fcb   153

 fcb    126  
 fcb   178
 
 fcb    127
 fcb   204
 
 fcb    128 
 fcb   230
 
 fcb    130 
 fcb   0
 
 fcb    131 
 fcb   0
 
 fcb    132 
 fcb   0
 
 fcb    133     220
 fcb   0

 fcb    134 
 fcb   0
 
 fcb    135 
 fcb   0 
 
 fcb    136 
 fcb   0
 
 fcb    137 
 fcb   0
 
 fcb    138     225
 fcb   0
 
 fcb    139 
 fcb   0
 
 fcb    140 
 fcb   0
 
 fcb    141 
 fcb   0
 
 fcb    142 
 fcb   0
 
 fcb    143     230
 fcb   0 

 fcb    144 
 fcb   0 
 
 fcb    145 
 fcb   0 
 
 fcb    146 
 fcb   0 
 
 fcb    147 
 fcb   0 
 
 fcb    148 
 fcb   0
 
 fcb    149 
 fcb   0
 
 fcb    150 
 fcb   0
 
 fcb    151 
 fcb   0

 fcb    152
 fcb   0 
 
 fcb    153     240
 fcb    0       

 fcb    154 
 fcb   0 
 
 fcb    155 
 fcb   0 
 
 fcb    156 
 fcb   0 
 
 fcb    157 
 fcb   0 
 
 fcb    158 
 fcb   0 
 
 fcb    159 
 fcb   0 
 
 fcb    160 
 fcb   0 
 
 fcb    161 
 fcb   0 
 
 fcb    162 
 fcb   0 
 
 fcb    163     250
 fcb    0       

 fcb    164 
 fcb   0 
 
 fcb    165 
 fcb   0 
 
 fcb    166 
 fcb   0 
 
 fcb    167 
 fcb   0 
 
 fcb    168     255
 fcb    0       
		
***************************************
*UHF

 org   62976 


 fcb    0
 fcb    0 

 fcb    0       
 fcb    25 

 fcb    0       
 fcb    51

 fcb    0       
 fcb    76

 fcb    0       
 fcb    102

 fcb    0       
 fcb    102

 fcb    0       
 fcb    127

 fcb    0       
 fcb    153

 fcb    0
 fcb    153

 fcb    0       
 fcb    178

 fcb    0       10
 fcb    178

 fcb    0 
 fcb    204

 fcb    0
 fcb    230

 fcb    0
 fcb    230

 fcb    1
 fcb    0

 fcb    1
 fcb    0

 fcb    1
 fcb    51

 fcb    1
 fcb    102

 fcb    1 
 fcb    127

 fcb    1
 fcb    153

 fcb    1       20
 fcb    204

 fcb    2
 fcb    0

 fcb    2       
 fcb    51

 fcb    2
 fcb    102

 fcb    2
 fcb    153

 fcb    2    25
 fcb    203

 fcb    3
 fcb    0 

 fcb    3
 fcb    76

 fcb    3
 fcb    127

 fcb    3
 fcb    178

 fcb    4    30
 fcb    0

 fcb    4
 fcb    76

 fcb    4
 fcb    127

 fcb    4
 fcb    178

 fcb    5
 fcb    0 

 fcb    5
 fcb    76

 fcb    5
 fcb    127

 fcb    5
 fcb    204

 fcb    6
 fcb    0

 fcb    6
 fcb    51

 fcb    6      40
 fcb    102

 fcb    6
 fcb    178

 fcb    7
 fcb    0

 fcb    7
 fcb    76

 fcb    7
 fcb    127

 fcb    7
 fcb    178

 fcb    8
 fcb    0

 fcb    8
 fcb    76

 fcb    8
 fcb    178

 fcb    9
 fcb   0 

 fcb    9      50
 fcb   127

 fcb    10
 fcb   0

 fcb    10
 fcb   127

 fcb    11
 fcb   0

 fcb    11
 fcb    76

 fcb    11
 fcb   153

 fcb    12
 fcb   0

 fcb    12
 fcb   76

 fcb    12
 fcb   178

 fcb    13
 fcb   0

 fcb    13    60
 fcb    76

 fcb    13
 fcb   178

 fcb    14
 fcb   0

 fcb    14
 fcb   76

 fcb    14
 fcb   178

 fcb    15
 fcb   0

 fcb    15
 fcb   76

 fcb    15
 fcb   178

 fcb    16
 fcb   0 

 fcb    16
 fcb   76

 fcb    16     70
 fcb    178

 fcb    17
 fcb   0 

 fcb    17
 fcb   127

 fcb    18
 fcb   0 

 fcb    18
 fcb   127

 fcb    19     75
 fcb   0 

 fcb    20
 fcb   0

 fcb    20
 fcb   127

 fcb    21
 fcb   0

 fcb    21
 fcb   127

 fcb    22    80
 fcb    0

 fcb    22
 fcb   127

 fcb    23
 fcb   0

 fcb    23
 fcb   127

 fcb    24
 fcb   0

 fcb    25
 fcb   0

 fcb    25
 fcb   127

 fcb    26
 fcb   0

 fcb    26
 fcb   127

 fcb    27
 fcb   0

 fcb    27    90
 fcb    127

 fcb    28
 fcb   0

 fcb    28
 fcb    127

 fcb    29
 fcb   0

 fcb    29
 fcb   127

 fcb    30
 fcb   0

 fcb    30
 fcb   127

 fcb    31
 fcb   0

 fcb    31
 fcb   127

 fcb    32
 fcb   0 

 fcb    32     100
 fcb   127

 fcb    33
 fcb   0

 fcb    33
 fcb   127

 fcb    34
 fcb   0

 fcb    34
 fcb   127

 fcb    35
 fcb   0

 fcb    35
 fcb   127

 fcb    36
 fcb   0

 fcb    36
 fcb   127

 fcb    37
 fcb   0

 fcb    37     110
 fcb    127

 fcb    38
 fcb   0

 fcb    39
 fcb   0

 fcb    39
 fcb   127

 fcb    40
 fcb   0

 fcb    41
 fcb   0

 fcb    41
 fcb   127

 fcb    42
 fcb   0

 fcb    42
 fcb   127

 fcb    43
 fcb   0

 fcb    43     120
 fcb    127

 fcb    44
 fcb  0

 fcb    44 
 fcb  127

 fcb    45 
 fcb   0 

 fcb    45 
 fcb   127

 fcb    46      125
 fcb   0
 
 fcb    46
 fcb   76

 fcb    46
 fcb   178

 fcb    47
 fcb   76

 fcb    47
 fcb    178

 fcb    48   130
 fcb    153

 fcb    49
 fcb   127

 fcb    50
 fcb   0

 fcb    51
 fcb   0

 fcb    51
 fcb   127

 fcb    52
 fcb   0

 fcb    53
 fcb   0

 fcb    53
 fcb   127

 fcb    54
 fcb   0

 fcb    55
 fcb   0

 fcb    56     140
 fcb    0

 fcb    56
 fcb    127

 fcb    57
 fcb    0

 fcb    58
 fcb    0

 fcb    59
 fcb    0

 fcb    59
 fcb    127

 fcb    60
 fcb    0

 fcb    61
 fcb    0

 fcb    61
 fcb    127

 fcb    62
 fcb    0

 fcb    63    150
 fcb    0

 fcb    63
 fcb   127

 fcb    64
 fcb   0

 fcb    65
 fcb    0

 fcb    65
 fcb   127

 fcb    66
 fcb   0

 fcb    67
 fcb   0

 fcb    68
 fcb   0

 fcb    68
 fcb   127

 fcb    69
 fcb   0

 fcb    70   160
 fcb    0

 fcb    71
 fcb   0

 fcb    71
 fcb   127

 fcb    72
 fcb   0

 fcb    73
 fcb   0

 fcb    73
 fcb    127

 fcb    74
 fcb   0 

 fcb    75
 fcb   0

 fcb    75
 fcb   127

 fcb    76
 fcb   0

 fcb    77   170
 fcb    0

 fcb    77
 fcb   127
 
 fcb    78
 fcb   0
 
 fcb   79
 fcb   0
 
 fcb    79
 fcb  127
 
 fcb    80   175
 fcb   0
 
 fcb    81
 fcb   0
 
 fcb    81
 fcb    127

 fcb    82
 fcb   0
 
 fcb    83
 fcb   0
 
 fcb    83  180
 fcb    127

 fcb    84
 fcb   0
 
 fcb    85
 fcb   0
 
 fcb    85
 fcb   127
 
 fcb    86
 fcb   0
 
 fcb    87
 fcb   0
 
 fcb    87
 fcb   127
 
 fcb    88
 fcb   0
 
 fcb    88
 fcb   127
 
 fcb    89
 fcb   0

 fcb    90     190
 fcb    0  

 fcb    91
 fcb   0
 
 fcb    91
 fcb   127
 
 fcb    92
 fcb   0
 
 fcb    93
 fcb   0
 
 fcb    94
 fcb   0
 
 fcb    95
 fcb   0 
 
 fcb    95
 fcb   127
 
 fcb    96
 fcb   0
 
 fcb    96
 fcb   127
 
 fcb    97   200
 fcb   0
 
 fcb    98
 fcb   0 

 fcb    99
 fcb   0 
 
 fcb    100
 fcb   0
 
 fcb    101
 fcb   0
 
 fcb    102
 fcb   0
 
 fcb    103
 fcb   0
 
 fcb    104
 fcb   0
 
 fcb    105
 fcb   0
 
 fcb    105
 fcb   127
 
 fcb    106   210
 fcb    0

 fcb    107
 fcb    0
 
 fcb    108
 fcb    0
 
 fcb    109
 fcb   0

 fcb    110
 fcb   0
 
 fcb    111
 fcb   0
 
 fcb    112
 fcb   0
 
 fcb    113
 fcb   0
 
 fcb    114
 fcb   0
 
 fcb    115
 fcb   0
 
 fcb    115   220
 fcb   127

 fcb    116
 fcb   0
 
 fcb    117
 fcb   0 
 
 fcb    118
 fcb   0
 
 fcb    119
 fcb   0
 
 fcb    119  225
 fcb   127
 
 fcb    120
 fcb   0
 
 fcb    121
 fcb   0
 
 fcb    122
 fcb   0
 
 fcb    123
 fcb   0
 
 fcb    124  230
 fcb   0 

 fcb    125
 fcb   0 
 
 fcb    126
 fcb   0 
 
 fcb    127
 fcb   0 
 
 fcb    128
 fcb   0 
 
 fcb    129
 fcb   0
 
 fcb    130
 fcb   0
 
 fcb    132
 fcb   0
 
 fcb    133
 fcb   0

 fcb    134
 fcb   0 
 
 fcb    135   240
 fcb    0       

 fcb    136
 fcb   0 
 
 fcb    137
 fcb   0 
 
 fcb    138
 fcb   0 
 
 fcb    139
 fcb   0 
 
 fcb    140
 fcb   0 
 
 fcb    141
 fcb   0 
 
 fcb    142
 fcb   0 
 
 fcb    143
 fcb   0 
 
 fcb    144
 fcb   0 
 
 fcb    145   250
 fcb    0       

 fcb    146
 fcb   0 
 
 fcb    147
 fcb   0 
 
 fcb    148
 fcb   0 
 
 fcb    149
 fcb   0 
 
 fcb    150   255
 fcb    0       
		

end

