/*
 * 915 MHz Wireless remote coax switch - Local unit
 * 
 * by Glen Popiel KW5GP
 * 
 * uses Adafruit Feather 32U4 LoRa
 * 
 * uses Adafruit ST7735 and GFX libraries, www.adafruit.com
 * uses Radiohead library by Mike McCauley, mikem@airspayce.com
 * uses Rotary Encoder library by 2014 by Matthias Hertel, www.mathertel.de
 * 
 */

#define debug_mode 0

#include <SPI.h>  // Built-in SPI library
#include <RH_RF95.h>  // Radiohead RF95 library
#include <Adafruit_GFX.h>    // Core graphics library
#include <Adafruit_ST7735.h> // Hardware-specific library
#include <RotaryEncoder.h>

// Define the ST7735 TFT Display pins
#define TFT_CS     10
#define TFT_RST    5  
#define TFT_DC     6
#define Backlight 11

// Simplify the TFT display color definitions
#define cyan  ST7735_CYAN
#define yellow  ST7735_YELLOW
#define black  ST7735_BLACK
#define green  ST7735_GREEN
#define white ST7735_WHITE
#define red ST7735_RED

//Define the Rotary Encoder pins
const int encoder_PinA = 18;  // Pin A of the Rotary Encoder (A0)
const int encoder_PinB = 19;  // Pin B of the Rotary Encoder (A1)
const int encoder_Switch_Pin = 2;  // Encoder Pushbutton Switch input pin (SDA - Digital Pin 2)

int encoder_Pos = 0; // The current encoder position
int old_encoder_Pos = 0; // The previous encoder position
int relay_select = 0; // Variable to hold the coax relay select value
int range_high = 4; // The max setting for the relay select option
int range_low = 0; // The minimum setting for the relay select option
boolean pushbutton; // Flag to indicate encoder pushbutton was pressed
boolean relay_change; // Flag to indicate it's time to set the selected coax relay
boolean switch_change;  // Flag to inidcate rotor encoder has been turned
String reply; // Variable to hold the reply data from the remote unit
String status_text; // Variable to hold the text displayed on the status line
char read_status; // The status of the data command (C = set relay confirmed, R = read, F = Rx failure, T = Timeout)
int relay_color = green;  // Variable to hold the color of the relay number displayed
int status_color = green; // Variable to hold the color of the status line
String relay_text = "off"; // Variable to hold the relay number display (0 = off)

// Instantiate the TFT Display
Adafruit_ST7735 tft = Adafruit_ST7735(TFT_CS, TFT_DC, TFT_RST);

// Instantiate the Rotary Encoder
RotaryEncoder Enc(encoder_PinA, encoder_PinB);

// Define the Feather32U4 pins
#define RFM95_CS 8
#define RFM95_RST 4
#define RFM95_INT 7

char radiopacket[3] = "  "; // Character array to hold the data packet to transmit
uint8_t buf[RH_RF95_MAX_MESSAGE_LEN]; // Receive data buffer - set to RF95 Maximum Message Length
uint8_t len = sizeof(buf);  // Variable to hold the size of the Rx buffer
int16_t packetnum = 0;  // packet counter, incremented with each xmission

// Define the radio frequency - must match remote's freq!
#define RF95_FREQ 915.0

// Instantiate the radio driver
RH_RF95 rf95(RFM95_CS, RFM95_INT);

void setup() 
{
  pinMode(RFM95_RST, OUTPUT); // Set the Pin Mode for the RF95's Reset pin
  
 // Set up the Rotary Encoder and enable the Internal Pull-up resistor on the Encoder Inputs 
  pinMode (encoder_PinA,INPUT_PULLUP);
  pinMode (encoder_PinB,INPUT_PULLUP);
  pinMode (encoder_Switch_Pin, INPUT_PULLUP);  
  
  attachInterrupt(encoder_Switch_Pin, read_pushbutton, LOW);  // Set up Interrupt 1 for the Rotary Encoder pushbutton switch
    
  digitalWrite(RFM95_RST, HIGH);  // Turn off RF95 Reset 

  if (debug_mode)
  {
    while (!Serial);
    Serial.begin(9600);
    delay(100);
    Serial.println("Resetting the Feather LoRa");
  }

  // Manually reset the RF95 radio
  digitalWrite(RFM95_RST, LOW);
  delay(10);
  digitalWrite(RFM95_RST, HIGH);
  delay(10);

  // Initialize the RF95 radio
  // Defaults after init are 434.0MHz, modulation GFSK_Rb250Fd250, +13dBm, Bw = 125 kHz, Cr = 4/5, Sf = 128chips/symbol, CRC on
  while (!rf95.init()) 
  {
    if (debug_mode)
    {
      Serial.println("LoRa radio init failed");
    }
    while (1);
  }

 if (debug_mode)
  {
    Serial.println("LoRa radio init OK!");
  }

  // Set the radio frequency 
  if (!rf95.setFrequency(RF95_FREQ)) 
  {
    if (debug_mode)
    {
      Serial.println("setFrequency failed");
    }
    while (1);
  }

  if (debug_mode)
  {
    Serial.println("Set Freq to: "); Serial.println(RF95_FREQ);
  }
  
  // Set the transmitter power output
  // The default transmitter power is 13dBm, using PA_BOOST.
  // If you are using RFM95/96/97/98 modules which uses the PA_BOOST transmitter pin, then 
  // you can set transmitter powers from 5 to 23 dBm:
  rf95.setTxPower(23, false);

  pinMode(Backlight, OUTPUT); // Set up the ST7735 TFT Display Backlight pin as output
  digitalWrite(Backlight, HIGH);  // Turn on the TFT Backlight

  // Initialize the ST7735 TFT Color LCD Display
  tft.initR(INITR_BLACKTAB);   // initialize a ST7735S chip, black tab

  tft.setRotation(1); // Rotate the TFT display 90 degrees Clockwise

  if (debug_mode)
  {
    Serial.println("TFT Initialized");
  }

  // Read the current remote status
  radiopacket[0]  = int('R'); // Read Status Command
  radiopacket[1]  = 0;  
  radiopacket[2] = 0;
  send_command(); // Transmite the command
  update_Display(); // Update the LCD display
  
}
    
void loop()
{
  read_encoder(); // Read the rotary encoder 
  if (switch_change)
  {
    // Rotary Encoder position has changed - change LCD color to cyan and wait for pushbutton press

    if (debug_mode)
    {
      Serial.println("Waiting for relay change command");
    }
    
    relay_color = cyan; // Set the relay text color to cyan
    status_color = cyan;  // Set the status text color to cyan
    status_text = "Change Pending"; // Change the status text to "Change Pending"
    update_Display(); // Update the LCD display
    switch_change = false;  // Clear the position changed flag
  } 

  if (pushbutton) // The Pushbutton Interrupt routine will set this flag if encoder pushbutton is pressed
  {
    pushbutton = false; // Turn off the pushbutton flag to indicate we have processed the Interrupt
    relay_change = true; // Set the relay_change flag
    status_color = yellow;  // Chage the status color to yellow to indicate we have sent the change relay command and waiting for reply
    status_text = "Changing"; // Change the status text to indicate we are changing the coax relay
    update_Display(); // Update the LCD display
    
    // Send the Set Relay command
    radiopacket[0] = 'S'; // Set Relay command - format = Sx where x is the relay number in ASCII
    radiopacket[1] = (relay_select + 48); // Add 48 to the relay number to convert it to an ASCII character
    send_command(); // Transmit the Command and wait for remote reply
    update_Display(); // Update the LCD display   
  }
}

// Function to update the LCD display
void update_Display()
{
  if (read_status == 'R') // We received a Read status command reply
  {
    relay_color = green;  // Change the relay text color to green
    status_color = green; // Change the status text color to green
    relay_select = int(reply[1] - 48);  // Convert the received relay position from ASCII to an integer
    status_text = "Read Ok  " + reply;  // Set the status line text to Read Ok and the received reply data
  } 
  
    if (read_status == 'C') // We received confirmation from a Set Relay command
    {
      relay_color = green;  // Change the relay text color to green
      status_color = green; // Change the status text color to green
      relay_select = int(reply[1]- 48);  // Convert the received relay position from ASCII to an integer
      status_text = "Set Command Ok  " + reply;  // Set the status line text to Set Command Ok and the received reply data
    }

    if(read_status == 'F')  // If we got a receive failure
    {
      relay_color = red;  // Change the relay text color to red
      status_color = red;  // Change the status text color to red
      status_text = "Command Failure";  // Set the status line text to Command Failure
    }

    if (read_status == 'T')  // If we got a receive timeout
    {
      relay_color = red;  // Change the relay text color to red
      status_color = red;  // Change the status text color to red
      status_text = "Command Timeout";  // Set the status line text to Command Failure
    }
    
    tft.fillScreen(black);  // Clear the LCD display
    tft.setCursor(45, 0);  // Set the cursor position for the relay indicator
    tft.setTextSize(2); // Set the Text size to a larger font
    tft.setTextColor(relay_color);  // Set relay indicator text color
    tft.println("Antenna");
    tft.setTextSize(4); // Set the Text size to a larger font
    if(relay_select == 0) // If the relay selected is 0, change the text to "Off"
    {
      tft.setCursor(50,40);
      relay_text = "Off";
    } else 
    {
      tft.setCursor(75,40);
      relay_text = String(relay_select);
    }
    
    tft.println(relay_text);  // Print the relay indicator

    if (debug_mode)
    {
      Serial.println("TFT done");
    }

    tft.setCursor(0, 120);  // Set the cursor position for the status line
    tft.setTextSize(0); // Set the font size to small
    tft.setTextColor(status_color); // Set the status text color
    tft.print("Status: "); 
    tft.print(status_text);
    
    read_status = ' ';  // Clear the read status after we display it
 
}
  
// Interrupt Function to read the rotary encoder pushbutton switch
void read_pushbutton()
{
  delay(500); // Wait 500ms for debounce
  if (!pushbutton) // No need to set flag again if pushbutton flag is set
  {
    if (debug_mode)
    {
      Serial.println("Pushbutton Interrupt");
    }
    pushbutton = true;  // Set the encoder pushbutton pressed flag
  }
} 

// Function to transmit data packets
void send_command()
{
  if (debug_mode)
  {
    Serial.println("Sending to rf95_server");
  }
  // radiopacket [0] and [1] are set beforehand
  radiopacket[2] = 0; // set [2] to 0 to indicate end of tx data

  if (debug_mode)
  {
    Serial.print("Sending "); 
    Serial.println(radiopacket);
    Serial.println("Sending..."); 
  }
  
  // Transmit the data
  delay(10);
  rf95.send((uint8_t *)radiopacket, 40);

  if (debug_mode)
  {
    Serial.println("Waiting for packet to complete..."); 
  }
  
  // Wait for a reply 
  delay(10);
  rf95.waitPacketSent(); // Wait for packet to be sent
  reply = "XXX"; // Fill reply with XXX so we know if we got data or not 
  
  if (debug_mode)
  {
    Serial.println("Waiting for reply...");
  } 
  
  delay(10);
  if (rf95.waitAvailableTimeout(1000))  // Wait for rx packet to be received
  { 
    // Should be a reply message for us now   
    if (rf95.recv(buf, &len)) // Check to see if there's data in the receive buffer
   {
      if (debug_mode)
      {
        Serial.print("Got reply: ");
      }
      
      reply = (char*)buf; // Copy the rx data to the reply string variable
      read_status = reply[0]; // Set the read status to the first character in the reply string
      
      if (debug_mode)
      {
        Serial.println((char*)buf);
        Serial.print("RSSI: ");
        Serial.println(rf95.lastRssi(), DEC);
      }    
    }
    else
    {
      if (debug_mode)
      {
        Serial.println("Receive failed");
      }
      
      read_status = 'F';  // Set read status to "F" if the receive failed
    }
  }
  else
  {
    if (debug_mode)
    {
      Serial.println("No reply, is there a listener around?");
    }
    read_status = 'T';  // Set the read status to "T" if we timed out waiting for data
  }
}

// Function to read the rotary encoder
void read_encoder()
{
  // Read the Encoder  
  Enc.tick(); // Required for library to check the encoder position
  encoder_Pos = Enc.getPosition(); // get the encoder position
  if (encoder_Pos != old_encoder_Pos) // If the Encoder has changed update selection 
  {
    if (debug_mode)
    {
      Serial.print("Encoder change: ");
      Serial.println(encoder_Pos);
    }
    
    if (encoder_Pos > old_encoder_Pos) // If we're increasing count
    {
      if (relay_select >= range_high) // Limit to top end of count
      {
        relay_select = range_high;
      } else 
      {
        relay_select = relay_select + 1 ;
      }
     
    } else 
    {
      if (relay_select <= range_low) // If we're decreasing count
      {
        relay_select = range_low; // Limit to lower end of count
      } else 
      {
        relay_select = relay_select - 1;
      }
    }

    old_encoder_Pos = encoder_Pos;  // Set the previous encoder position to the current position
    switch_change = true; // Turn on flag to indicate rotary encoder position has changed
    
    if (debug_mode)
    {
      Serial.print("Switch change: ");
      Serial.println(relay_select);
    }
  }
}

